"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.areEqual = exports.luminanceFrom = exports.dynamicWith = exports.named = exports.rgbWith = exports.htmlWith = void 0;
const optional_1 = require("../types/optional");
// endregion
// region Constructors
/**
 * Create new `HTMLColor` from hexadecimal string representation.
 *
 * @param hexString - Hexadecimal string representation.
 */
function htmlWith(hexString) {
    if (optional_1.isNothing(hexString)) {
        return null;
    }
    return {
        $kind: "html",
        value: hexString,
    };
}
exports.htmlWith = htmlWith;
/**
 * Create new `RBGColor` with RGB components and opacity value.
 *
 * @param red - Red color value.
 * @param green - Green color value.
 * @param blue - Blue color value.
 * @param alpha - Opacity value.
 */
function rgbWith(red, green, blue, alpha = 1.0) {
    const newColor = {
        $kind: "rgb",
        red: red,
        green: green,
        blue: blue,
        alpha: alpha,
    };
    return newColor;
}
exports.rgbWith = rgbWith;
/**
 * Create new named color using the color name.
 *
 * @param name - The name of the color.
 */
function named(name) {
    const newColor = {
        $kind: "named",
        name: name,
    };
    return newColor;
}
exports.named = named;
/**
 * Create new dynamic color with light and dark color variants.
 *
 * @param lightColor - The light color variant.
 * @param lightHighContrastColor - The light hight-contrast color variant.
 * @param darkColor - The dark color variant.
 * @param darkHighContrastColor - The dark hight-contrast color variant.
 */
function dynamicWith(lightColor, lightHighContrastColor, darkColor, darkHighContrastColor) {
    const newColor = {
        $kind: "dynamic",
        lightColor: lightColor,
        lightHighContrastColor: lightHighContrastColor,
        darkColor: darkColor,
        darkHighContrastColor: darkHighContrastColor,
    };
    return newColor;
}
exports.dynamicWith = dynamicWith;
// endregion
// region Properties
/**
 * Get the luminance of the color.
 *
 * @param rgbColor - The RGB color to get luminance for.
 */
function luminanceFrom(rgbColor) {
    // Note: This is lifted from UIColor_Private
    // Using RGB color components, calculates and returns (0.2126 * r) + (0.7152 * g) + (0.0722 * b).
    return rgbColor.red * 0.2126 + rgbColor.green * 0.7152 + rgbColor.blue * 0.0722;
}
exports.luminanceFrom = luminanceFrom;
// endregion
// region Identity
/**
 * Compare two colors for equality.
 *
 * @param color1 - Left hand side color to compare.
 * @param color2 - Right hand side color to compare.
 * @returns A Boolean indicating whether the colors are equal.
 */
function areEqual(color1, color2) {
    if (optional_1.isNothing(color1)) {
        return optional_1.isNothing(color2);
    }
    else if (optional_1.isNothing(color2)) {
        return optional_1.isNothing(color1);
    }
    const kind1 = color1.$kind;
    const kind2 = color2.$kind;
    if (kind1 === "named" && kind2 === "named") {
        const namedColor1 = color1;
        const namedColor2 = color2;
        return namedColor1.name === namedColor2.name;
    }
    else if (kind1 === "rgb" && kind2 === "rgb") {
        const rgbColor1 = color1;
        const rgbColor2 = color2;
        return (rgbColor1.red === rgbColor2.red &&
            rgbColor1.green === rgbColor2.green &&
            rgbColor1.blue === rgbColor2.blue &&
            rgbColor1.alpha === rgbColor2.alpha);
    }
    else if (kind1 === "dynamic" && kind2 === "dynamic") {
        const dynamicColor1 = color1;
        const dynamicColor2 = color2;
        return (areEqual(dynamicColor1.lightColor, dynamicColor2.lightColor) &&
            areEqual(dynamicColor1.lightHighContrastColor, dynamicColor2.lightHighContrastColor) &&
            areEqual(dynamicColor1.darkColor, dynamicColor2.darkColor) &&
            areEqual(dynamicColor1.darkHighContrastColor, dynamicColor2.darkHighContrastColor));
    }
    else {
        return false;
    }
}
exports.areEqual = areEqual;
// endregion
//# sourceMappingURL=color.js.map