"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeArtworkURLTemplate = void 0;
const validation = require("../json/validation");
const optional_1 = require("../types/optional");
const urls_1 = require("../util/urls");
/**
 * Regex to parse artwork URL template string.
 */
const URL_TEMPLATE_PARSER = new RegExp("^({w}|[0-9]+(?:.[0-9]*)?)x({h}|[0-9]+(?:.[0-9]*)?)({c}|[a-z]{2}).({f}|[a-z]+)$");
/**
 * Create an instance of artwork URL template from string.
 * @param fromString - String to create artwork URL template from.
 * @returns A new artwork URL template or `null` if string
 * does not represent a valid artwork URL template.
 */
function makeArtworkURLTemplate(fromString) {
    // A valid URL that ends with '{w}x{h}{c}.{f}'
    // with any of placeholders possibly resolved to an actual value.
    const url = new urls_1.URL(fromString);
    if (url.pathname === undefined) {
        validation.context("makeArtworkURLTemplate", () => {
            validation.unexpectedType("ignoredValue", "A valid URL string", fromString);
        });
        return null;
    }
    // Expecting 5 matches: whole string + width, height, crop code and format.
    const lastPathComponent = fromString.substring(fromString.lastIndexOf("/") + 1);
    const matches = URL_TEMPLATE_PARSER.exec(lastPathComponent);
    if (optional_1.isNothing(matches) || matches.length !== 5) {
        validation.context("makeArtworkURLTemplate", () => {
            validation.unexpectedType("ignoredValue", "A valid artwork URL template ending with {w}x{h}{c}.{f} format", lastPathComponent);
        });
        return null;
    }
    return fromString;
}
exports.makeArtworkURLTemplate = makeArtworkURLTemplate;
//# sourceMappingURL=artwork.js.map