"use strict";
/* eslint-disable @typescript-eslint/no-extraneous-class */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LazyProxyFactory = void 0;
// LazyProxyFactory creates a Proxy which lazily loads its target once needed
class LazyProxyFactory {
    static makeProxy(name, lazyProvider) {
        const target = new ProxyTarget(lazyProvider);
        const handler = {
            get: (_, property, __) => {
                // Create the lazy dependency if needed
                if (target.createRealObjectIfNeeded(name) === undefined) {
                    // The dependency unexpectedly couldn't be created, so we have nothing to proxy to
                    return undefined;
                }
                // Get the property, which could be a function
                const res = target.realObject[property];
                // If it's a function, bind it to the real object
                if (typeof res == "function") {
                    return res.bind(target.realObject);
                }
                return res;
            },
            set: (_, property, value) => {
                if (target.createRealObjectIfNeeded(name) === undefined) {
                    return false;
                }
                return Reflect.set(target.realObject, property, value, target.realObject);
            },
            has: (_, property) => {
                if (target.createRealObjectIfNeeded(name) === undefined) {
                    return false;
                }
                return Reflect.has(target.realObject, property);
            },
            deleteProperty: (_, property) => {
                if (target.createRealObjectIfNeeded(name) === undefined) {
                    return false;
                }
                return Reflect.deleteProperty(target.realObject, property);
            },
            ownKeys: (_) => {
                if (target.createRealObjectIfNeeded(name) === undefined) {
                    return [];
                }
                return Reflect.ownKeys(target.realObject);
            },
            getOwnPropertyDescriptor: (_, property) => {
                if (target.createRealObjectIfNeeded(name) === undefined) {
                    return undefined;
                }
                return Reflect.getOwnPropertyDescriptor(target.realObject, property);
            },
            defineProperty: (_, property, descriptor) => {
                if (target.createRealObjectIfNeeded(name) === undefined) {
                    return false;
                }
                return Reflect.defineProperty(target.realObject, property, descriptor);
            },
            preventExtensions: (_) => {
                if (target.createRealObjectIfNeeded(name) === undefined) {
                    return false;
                }
                /* We need to prevent extensions on the actual proxy target
                otherwise the Proxy throws a TypeError */
                Reflect.preventExtensions(target);
                return Reflect.preventExtensions(target.realObject);
            },
            isExtensible: (_) => {
                if (target.createRealObjectIfNeeded(name) === undefined) {
                    return false;
                }
                return Reflect.isExtensible(target.realObject);
            },
            getPrototypeOf: (_) => {
                if (target.createRealObjectIfNeeded(name) === undefined) {
                    return null;
                }
                return Reflect.getPrototypeOf(target.realObject);
            },
            setPrototypeOf: (_, prototype) => {
                if (target.createRealObjectIfNeeded(name) === undefined) {
                    return false;
                }
                return Reflect.setPrototypeOf(target.realObject, prototype);
            },
        };
        const res = new Proxy(target, handler);
        return res;
    }
}
exports.LazyProxyFactory = LazyProxyFactory;
/* The proxy target begins by holding nothing,
and eventually holds the real proxy target once needed */
class ProxyTarget {
    constructor(lazyProvider) {
        this.lazyProvider = lazyProvider;
    }
    createRealObjectIfNeeded(name) {
        if (this.realObject === undefined) {
            this.realObject = this.lazyProvider.lazyDependencyFor(name);
        }
        return this.realObject;
    }
}
//# sourceMappingURL=jet-proxy.js.map