/**
 * Created by joel on 1/25/18.
 */
import * as serverData from "../json-parsing/server-data";
import { ResponseMetadata } from "../network/network";
export function dataFromDataContainer(objectGraph, dataContainer) {
    const dataArray = serverData.asArrayOrEmpty(dataContainer, "data");
    if (dataArray.length > 1) {
        objectGraph.console.warn("tried to extract data from container but more than one member present");
    }
    if (dataArray.length !== 1) {
        return null;
    }
    return dataArray[0];
}
export function dataCollectionFromDataContainer(dataContainer) {
    return serverData.asArrayOrEmpty(dataContainer, "data");
}
/**
 * Check whether or not a server vended `Data` object is hydrated or not.
 * @param {Data} data to check if hydrated.
 */
export function isDataHydrated(data) {
    return serverData.isDefinedNonNull(data.attributes);
}
/**
 * Check whether or not a entire data collection contains elements that are fully hydrated.
 * @param {Data[]} dataArray Data array to check.
 */
export function isDataCollectionHydrated(dataCollection) {
    // Iterate from the back to determine if fully hydrated faster - unhydrated elements tend to be latter elements.
    const lastIndex = dataCollection.length - 1;
    for (let index = lastIndex; index >= 0; index--) {
        const data = dataCollection[index];
        if (!isDataHydrated(data)) {
            return false;
        }
    }
    return true;
}
/**
 * Check whether or not a entire data collection at least has 1 element that is fully hydrated.
 * @param {Data[]} dataArray Data array to check.
 */
export function isDataCollectionPartiallyHydrated(dataCollection) {
    for (const data of dataCollection) {
        if (isDataHydrated(data)) {
            return true;
        }
    }
    return false;
}
/**
 * Check whether or not a today data module represents a today page on the heuristic that:
 * - The "Today" modules have labels with value 'TodayForApps' as opposed to `WhatYouMissed`
 * - Date marker "Today" modules have a 'date' field.
 */
export function isModuleTodayForApps(todayModule) {
    const todayModuleId = "TodayForApps";
    const date = serverData.traverse(todayModule, "date");
    return todayModule.label === todayModuleId || serverData.isDefinedNonNull(date);
}
/**
 * Get chart results from a server response. Always returns chart segments with valid data
 */
export function chartResultsFromServerResponse(response) {
    const resultsArray = serverData.asArrayOrEmpty(response, "results.apps");
    return resultsArray.filter((segment) => {
        return !serverData.isNull(segment.data);
    });
}
export function dataCollectionFromResultsListContainer(resultsListContainer) {
    return serverData.asArrayOrEmpty(resultsListContainer, "results.contents");
}
/**
 * Get the metrics dictionary included in the meta data of a mediaApi Data object.
 * @param {MetaDataProviding} metaDataProvidingObject
 * @returns {MapLike<string> | null}
 */
export function metricsFromMediaApiObject(metaDataProvidingObject) {
    return serverData.asDictionary(metaDataProvidingObject, "meta.metrics");
}
// endregion
//# sourceMappingURL=data-structure.js.map