import { withActiveIntent } from "../../foundation/dependencies/active-intent";
import { injectSEOData } from "../../api/models/web-renderable-page";
import { buildArticlePageRequest } from "../../common/today/article-request";
import { fetchData } from "../../foundation/media/network";
import { articlePageFromResponse, ArticleParseContext } from "../../common/today/article";
import { routableArticlePageWithPlatformRoutes, routableArticlePageWithoutPlatformRoutes, makeRoutableArticlePageCanonicalUrl, } from "../../common/today/routable-article-page-url-utils";
import { injectWebNavigation } from "../../common/web-navigation/inject-web-navigation";
import { inferPreviewPlatformFromEditorialPlatforms } from "../../common/preview-platform";
import { validateAdamId } from "../../foundation/media/util";
export const RoutableArticlePageIntentController = {
    $intentKind: "RoutableArticlePageIntent",
    routes(objectGraph) {
        return [
            // Note: order here is important! Both URL patterns provide an option where there is a
            // single dynamic segment in front of the stable `/story` segment; it could be either
            // the platform without an explicit storefront, or a storefront without an explicit platform.
            // We want to assume the latter, so the URLs without a platform must be defined first
            ...routableArticlePageWithoutPlatformRoutes(objectGraph),
            ...routableArticlePageWithPlatformRoutes(objectGraph),
        ];
    },
    async perform(intent, objectGraphWithoutActiveIntent) {
        return await withActiveIntent(objectGraphWithoutActiveIntent, intent, async (objectGraph) => {
            var _a, _b, _c;
            // See: https://github.pie.apple.com/its/Jingle/blob/ce14e21b6ac3dd4be884aa12b26d4e79c6d8aa7a/MZStorePlatform/src/main/java/com/apple/jingle/store/mediaapi/resource/SFMediaAPICommonResourceType.java#L89
            validateAdamId(objectGraph, intent.id);
            const mediaApiRequest = buildArticlePageRequest(objectGraph, intent, false);
            const response = await fetchData(objectGraph, mediaApiRequest);
            // If the user visits a "story" without an explicit "platform" specified, we need to infer a
            // desirable default so that we can look up the correct platform attributes when rendering
            // the view-model
            const previewPlatform = (_a = intent.platform) !== null && _a !== void 0 ? _a : inferPreviewPlatformFromEditorialPlatforms(objectGraph, response);
            if (!objectGraph.activeIntent.previewPlatform) {
                objectGraph.activeIntent.setInferredPreviewPlatform(previewPlatform);
            }
            const articleParseContext = new ArticleParseContext();
            const page = articlePageFromResponse(objectGraph, response, articleParseContext);
            // The Media API-provided `.canonicalURL` will not contain the explicit `{platform}`
            // segment, if required; we need to override their value to make use of the URL
            // factory provided by our routing utils instead
            page.canonicalURL = makeRoutableArticlePageCanonicalUrl(objectGraph, intent);
            if (objectGraph.client.isWeb) {
                const webNavigation = injectWebNavigation(objectGraph, page, previewPlatform);
                if ((_b = page.card) === null || _b === void 0 ? void 0 : _b.title) {
                    webNavigation.title = page.card.title;
                }
                injectSEOData(page, (_c = objectGraph.seo) === null || _c === void 0 ? void 0 : _c.getSEODataForArticlePage(objectGraph, page, response));
            }
            return page;
        });
    },
};
//# sourceMappingURL=routable-article-page-intent-controller.js.map