import * as validation from "@jet/environment/json/validation";
import { TodayCardMediaAppIcon, TodayCardMediaList, TodayCardParagraphOverlay } from "../../../api/models";
import { isDefinedNonNull } from "../../../foundation/json-parsing/server-data";
import { deepLinkUrlFromData } from "../../linking/external-deep-link";
import { applyTodayCardOverridesForAcquisitionStoryIfNecessary, relatedContentOverlayFromData, } from "../today-card-overlay-util";
import { cardDisplayStyleFromData, lockupsForRelatedContent, offerEnvironmentForTodayCard, offerStyleForTodayCard, popTodayCardLocation, pushTodayCardLocation, relatedCardContentsContentsFromData, } from "../today-card-util";
import { addArtworkMediaToBaseCard, createTodayBaseCard } from "./today-base-card-builder";
/**
 * Create TodayCard displaying the SingleApp card display style
 *
 * @param objectGraph The dependency graph for the App Store
 * @param data The media api data to build the card from
 * @param cardConfig The configuration for the card
 * @param context The parse context for the over all today page
 * @param augmentingData that stores some additional responses that may be used to enhance the contents of `data`
 * @returns The newly created TodayCard, either artwork or the single app icon.
 */
export function createTodaySingleAppCard(objectGraph, data, cardConfig, context, augmentingData) {
    return validation.context("createTodaySingleAppCard", () => {
        const singleAppCard = createTodayBaseCard(objectGraph, data, cardConfig, context);
        pushTodayCardLocation(objectGraph, data, cardConfig, context);
        // Next prefer artwork card, and then fallback to substyles
        const didAddArtworkToCard = addArtworkMediaToBaseCard(objectGraph, data, singleAppCard, cardConfig, context);
        const relatedContent = relatedCardContentsContentsFromData(objectGraph, data);
        const relatedContentLockups = lockupsForRelatedContent(objectGraph, relatedContent, cardConfig, context.pageInformation, context.locationTracker, "todayCard", null, deepLinkUrlFromData(objectGraph, data));
        const overlay = relatedContentOverlayFromData(objectGraph, singleAppCard, cardConfig, data, relatedContentLockups);
        if (isDefinedNonNull(overlay) && overlay instanceof TodayCardParagraphOverlay) {
            overlay.style = "white";
        }
        singleAppCard.overlay = overlay;
        if (!didAddArtworkToCard && relatedContent.length === 1) {
            const didAddSingleAppFallback = addSingleAppFallbackToCard(objectGraph, data, singleAppCard, relatedContent, cardConfig, context);
            if (!didAddSingleAppFallback) {
                popTodayCardLocation(context);
                return null;
            }
        }
        // Special post-processing step for Acquisition story cards.
        // This is needed to splice in data not included in initial response.
        applyTodayCardOverridesForAcquisitionStoryIfNecessary(objectGraph, singleAppCard, cardConfig, cardDisplayStyleFromData(data, cardConfig.coercedCollectionTodayCardDisplayStyle), data, augmentingData, context);
        popTodayCardLocation(context);
        return singleAppCard;
    });
}
/**
 * Adds a single app fallback to the card.
 *
 * @param objectGraph - The object graph of the app store.
 * @param data - The media data.
 * @param card - The today card.
 * @param relatedContent - The related content data.
 * @param cardConfig - The configuration for the today card.
 * @param context - The parse context for today card.
 * @returns Returns true or false based on whether we could create a single app fallback.
 */
export function addSingleAppFallbackToCard(objectGraph, data, card, relatedContent, cardConfig, context) {
    if (relatedContent.length !== 1) {
        return false;
    }
    cardConfig.useJoeColorIconPlaceholder = true;
    card.style = "dark";
    const offerStyle = offerStyleForTodayCard(objectGraph, card.style);
    const relatedContentLockups = lockupsForRelatedContent(objectGraph, relatedContent, cardConfig, context.pageInformation, context.locationTracker, offerEnvironmentForTodayCard(card.style), offerStyle, deepLinkUrlFromData(objectGraph, data));
    if (relatedContentLockups.length !== 1) {
        return false;
    }
    const lockup = relatedContentLockups[0];
    // tvOS doesn't have layout to properly render appIcon media.
    // However, `media` can't be simply removed from TodayCard, as it is a required field in native code.
    // This `TodayCardMediaList` works as an empty media placeholder,
    // that allows to render text from TodayCard but not the image.
    // From StarlightB AppIcon is disabled in native code and this check can be removed in the next major release.
    if (objectGraph.client.isTV && !objectGraph.props.enabled("disableAppIconMediaTodayHeader")) {
        card.media = new TodayCardMediaList([], [], [], [], undefined, undefined);
    }
    else {
        card.media = new TodayCardMediaAppIcon(lockup.icon);
    }
    return true;
}
//# sourceMappingURL=today-single-app-card-builder.js.map