import * as validation from "@jet/environment/json/validation";
import { TodayCardMediaList } from "../../../api/models";
import * as videoDefaults from "../../constants/video-constants";
import { todayCardArtworkDetails } from "../artwork/today-artwork-util";
import { applyTodayCardOverridesForAcquisitionStoryIfNecessary } from "../today-card-overlay-util";
import { applyMultiAppFallbackToCollectionCard, cardDisplayStyleFromData, cardStyleFromJoeColorsWithoutFallback, listFallbackLimit, lockupsForRelatedContent, popTodayCardLocation, pushTodayCardLocation, relatedCardContentsContentsFromData, } from "../today-card-util";
import { TodayCardDisplayStyle, } from "../today-types";
import { createTodayBaseCard } from "./today-base-card-builder";
import { isEligibleForGamesApp } from "../../content/content";
/**
 * Create TodayCard displaying the list of lockups
 *
 * @param objectGraph The dependency graph for the App Store
 * @param data The media api data to build the card from
 * @param cardConfig The configuration for the card
 * @param context The parse context for the over all today page
 * @param augmentingData that stores some additional responses that may be used to enhance the contents of `data`
 * @returns The newly created TodayCard, using a list or numbered list media.
 */
export function createTodayListCard(objectGraph, data, cardConfig, context, augmentingData) {
    return validation.context("createTodayListCard", () => {
        var _a, _b, _c;
        const listCard = createTodayBaseCard(objectGraph, data, cardConfig, context);
        pushTodayCardLocation(objectGraph, data, cardConfig, context);
        cardConfig.useJoeColorIconPlaceholder = true;
        let relatedContents = relatedCardContentsContentsFromData(objectGraph, data);
        if (preprocessor.GAMES_TARGET) {
            // Filtering out non-eligible games as they can be used as content for stories collection.
            // see `createFallbackListShelf` in article.ts.
            relatedContents = relatedContents.filter((item) => isEligibleForGamesApp(item));
        }
        const listLockups = lockupsForRelatedContent(objectGraph, relatedContents, cardConfig, context.pageInformation, context.locationTracker, undefined, undefined, undefined);
        if (listLockups.length === 0) {
            popTodayCardLocation(context);
            return null;
        }
        cardConfig.canDisplayArcadeOfferButton = false;
        // We require at least `listFallbackLimit` lockups for the list card
        // If this is for use on top of an article its ok to have less than `listFallbackLimit`
        if (listLockups.length < listFallbackLimit(objectGraph) && cardConfig.enableListCardToMultiAppFallback) {
            applyMultiAppFallbackToCollectionCard(objectGraph, data, listLockups, listCard);
        }
        else {
            const cardDisplayStyle = cardDisplayStyleFromData(data, cardConfig.coercedCollectionTodayCardDisplayStyle);
            if (cardDisplayStyle === TodayCardDisplayStyle.NumberedList) {
                let index = 1;
                for (const lockup of listLockups) {
                    lockup.ordinal = `${index}`;
                    index++;
                }
            }
            const mediaDetails = todayCardArtworkDetails(objectGraph, data, cardConfig);
            const cardStyle = cardStyleFromJoeColorsWithoutFallback(mediaDetails === null || mediaDetails === void 0 ? void 0 : mediaDetails.joeColors, "bgColor");
            const isModernListCard = objectGraph.host.isiOS;
            // Since list videos are very wide (21:9) and ~50% masked, we prevent them from going fullscreen
            mediaDetails === null || mediaDetails === void 0 ? void 0 : mediaDetails.videos.forEach((video) => {
                video.canPlayFullScreen = false;
                video.playbackControls = videoDefaults.noControls(objectGraph);
                video.autoPlayPlaybackControls = videoDefaults.noControls(objectGraph);
            });
            listCard.style = isModernListCard ? undefined : "white";
            listCard.media = new TodayCardMediaList(listLockups, (_a = mediaDetails === null || mediaDetails === void 0 ? void 0 : mediaDetails.artworks) !== null && _a !== void 0 ? _a : [], (_b = mediaDetails === null || mediaDetails === void 0 ? void 0 : mediaDetails.videos) !== null && _b !== void 0 ? _b : [], (_c = mediaDetails === null || mediaDetails === void 0 ? void 0 : mediaDetails.artworkLayoutsWithMetrics) !== null && _c !== void 0 ? _c : [], undefined, isModernListCard ? cardStyle !== "white" : cardStyle === "dark");
        }
        // Special post-processing step for Acquisition story cards.
        // This is needed to splice in data not included in initial response.
        applyTodayCardOverridesForAcquisitionStoryIfNecessary(objectGraph, listCard, cardConfig, cardDisplayStyleFromData(data, cardConfig.coercedCollectionTodayCardDisplayStyle), data, augmentingData, context);
        popTodayCardLocation(context);
        return listCard;
    });
}
//# sourceMappingURL=today-list-card-builder.js.map