import * as validation from "@jet/environment/json/validation";
import { isSome } from "@jet/environment/types/optional";
import { TodayCardMediaBrandedSingleApp } from "../../../api/models";
import { isNullOrEmpty } from "../../../foundation/json-parsing/server-data";
import { deepLinkUrlFromData } from "../../linking/external-deep-link";
import { todayCardArtworkDetails } from "../artwork/today-artwork-util";
import { applyTodayCardOverridesForAcquisitionStoryIfNecessary, relatedContentOverlayFromData, } from "../today-card-overlay-util";
import { cardDisplayStyleFromData, cardStyleFromJoeColors, lockupsForRelatedContent, offerEnvironmentForTodayCard, offerStyleForTodayCard, popTodayCardLocation, pushTodayCardLocation, relatedCardContentsContentsFromData, todayCardArtworkTitleBackingGradientForKey, } from "../today-card-util";
import { createTodayBaseCard } from "./today-base-card-builder";
import { createTodaySingleAppCard } from "./today-single-app-card-builder";
/**
 * Create TodayCard displaying the App of the Day and Game of the Day editorial items
 *
 * @param objectGraph The dependency graph for the App Store
 * @param data The media api data to build the card from
 * @param cardConfig The configuration for the card
 * @param context The parse context for the over all today page
 * @param augmentingData that stores some additional responses that may be used to enhance the contents of `data`
 * @returns The newly created TodayCard, using an artwork media to display the App of the Day and Game of the Day
 */
export function createTodayBrandedCard(objectGraph, data, cardConfig, context, augmentingData) {
    return validation.context("createTodayBrandedCard", () => {
        const brandedCard = createTodayBaseCard(objectGraph, data, cardConfig, context);
        pushTodayCardLocation(objectGraph, data, cardConfig, context);
        // Configure the card style
        const mediaDetails = todayCardArtworkDetails(objectGraph, data, cardConfig);
        if (isSome(mediaDetails) &&
            isSome(mediaDetails.joeColors.bgColor) &&
            (mediaDetails.artworks.length > 0 || mediaDetails.videos.length > 0)) {
            brandedCard.style = cardStyleFromJoeColors(mediaDetails === null || mediaDetails === void 0 ? void 0 : mediaDetails.joeColors, "bgColor");
        }
        else {
            // If we don't have artwork, try building a single app card instead.
            popTodayCardLocation(context);
            return createTodaySingleAppCard(objectGraph, data, cardConfig, context);
        }
        const offerStyle = offerStyleForTodayCard(objectGraph, brandedCard.style);
        const relatedContent = relatedCardContentsContentsFromData(objectGraph, data);
        const relatedContentLockups = lockupsForRelatedContent(objectGraph, relatedContent, cardConfig, context.pageInformation, context.locationTracker, offerEnvironmentForTodayCard(brandedCard.style), offerStyle, deepLinkUrlFromData(objectGraph, data), true);
        if (isNullOrEmpty(relatedContentLockups) || relatedContentLockups.length !== 1) {
            popTodayCardLocation(context);
            return null;
        }
        if (!cardConfig.isHorizontalShelfContext) {
            brandedCard.overlay = relatedContentOverlayFromData(objectGraph, brandedCard, cardConfig, data, relatedContentLockups);
        }
        // Configure the card media
        const brandedAppLockup = relatedContentLockups[0];
        brandedCard.media = new TodayCardMediaBrandedSingleApp(brandedAppLockup.icon, mediaDetails.artworks, mediaDetails.videos, mediaDetails.artworkLayoutsWithMetrics, todayCardArtworkTitleBackingGradientForKey(objectGraph, data, cardConfig));
        brandedCard.media.impressionMetrics = brandedAppLockup.impressionMetrics;
        // Special post-processing step for Acquisition story cards.
        // This is needed to splice in data not included in initial response.
        applyTodayCardOverridesForAcquisitionStoryIfNecessary(objectGraph, brandedCard, cardConfig, cardDisplayStyleFromData(data, cardConfig.coercedCollectionTodayCardDisplayStyle), data, augmentingData, context);
        popTodayCardLocation(context);
        return brandedCard;
    });
}
//# sourceMappingURL=today-branded-card-builder.js.map