/**
 * Build methods for Search Spell Correction.
 * Spell correction allows users to change an previous search term to higher or lower confidence searches.
 */
import * as models from "../../api/models";
import { isDefinedNonNull } from "../../foundation/json-parsing/server-data";
import { addEventsToSearchAction } from "../metrics/helpers/clicks";
import { popLocation, pushBasicLocation } from "../metrics/helpers/location";
/**
 * Build a `SearchResultsMessage` for a search result with given `SearchTermContext`
 * @param termContext Context for the state of search terms.
 */
export function spellCorrectionMessageFromTermContext(objectGraph, termContext, metricsOptions) {
    const showUndoSpellCorrection = isDefinedNonNull(termContext.correctedTerm);
    const showSuggestion = isDefinedNonNull(termContext.suggestedTerm);
    pushBasicLocation(objectGraph, {
        pageInformation: metricsOptions.pageInformation,
        locationTracker: metricsOptions.locationTracker,
        targetType: "link",
    }, "spellCorrection");
    if (showUndoSpellCorrection) {
        return undoSpellCorrectionMessageFromTermContext(objectGraph, termContext, metricsOptions);
    }
    if (showSuggestion) {
        return correctionSuggestionMessageFromTermContext(objectGraph, termContext, metricsOptions);
    }
    popLocation(metricsOptions.locationTracker);
    return null;
}
/**
 * Build a message that allows user to undo an auto-applied spell correction on user-initiated term.
 * This is the "Showing Results for ABC. Search Instead For XYZ" variation for high-confidence misspellings
 * @param termContext Context for the state of search terms.
 * @param locationTracker Location tracker for page it is appearing in.
 */
function undoSpellCorrectionMessageFromTermContext(objectGraph, termContext, metricsOptions) {
    // SearchAction for `term` again, disabling spell correction to `correctedTerm`.
    const uncorrectedTerm = termContext.term;
    const searchInsteadAction = searchActionForSpellCorrection(objectGraph, uncorrectedTerm, termContext.resultsTerm, "undoSpellCorrection");
    addEventsToSearchAction(objectGraph, searchInsteadAction, "button", metricsOptions.locationTracker);
    // "Showing Results for <i>correctedTerm</i>" with no links.
    const correctedToTerm = `<i>${termContext.correctedTerm}</i>`;
    const showingResultsForTemplate = objectGraph.loc.string("SEARCH_SHOWING_RESULTS_FOR_TEMPLATE");
    const showingResultsForMessage = showingResultsForTemplate.replace("{searchTerm}", correctedToTerm);
    const primaryText = new models.LinkableText(new models.StyledText(showingResultsForMessage, "text/x-apple-as3-nqml"), {});
    // "Search Instead for term" with link.
    const searchInsteadForTemplate = objectGraph.loc.string("SEARCH_SEARCH_INSTEAD_FOR_TEMPLATE");
    const searchInsteadForMessage = searchInsteadForTemplate.replace("{searchTerm}", uncorrectedTerm);
    const searchInsteadForLinks = {};
    searchInsteadForLinks[`${searchInsteadForMessage}`] = searchInsteadAction;
    const secondaryText = new models.LinkableText(new models.StyledText(searchInsteadForMessage), searchInsteadForLinks);
    return new models.SearchResultsMessage(primaryText, secondaryText, searchInsteadAction);
}
/**
 * Build a message that allows user to accept a suggestion to modify a user-initiated term.
 * This is the "Did you mean ABC?" variation for low-confidence misspellings
 * @param termContext Context for the state of search terms.
 * @param locationTracker Location tracker for page it is appearing in.
 */
function correctionSuggestionMessageFromTermContext(objectGraph, termContext, metricsOptions) {
    // Search action for `suggestedTerm`
    const suggestedTerm = termContext.suggestedTerm;
    const suggestedSearchAction = searchActionForSpellCorrection(objectGraph, suggestedTerm, termContext.resultsTerm, "applySpellCorrection");
    addEventsToSearchAction(objectGraph, suggestedSearchAction, "button", metricsOptions.locationTracker);
    // "Did you mean <i>suggestedTerm</i>?" where suggested term is linked.
    const styledSuggestedTerm = `<i>${suggestedTerm}</i>`;
    const didYouMeanTemplate = objectGraph.loc.string("SEARCH_DID_YOU_MEAN_TEMPLATE");
    const didYouMeanMessage = didYouMeanTemplate.replace("{searchTerm}", styledSuggestedTerm);
    // Link both the raw suggested term, and suggested term followed by ?
    const didYouMeanLinks = {};
    didYouMeanLinks[`${suggestedTerm}`] = suggestedSearchAction;
    didYouMeanLinks[`${suggestedTerm}?`] = suggestedSearchAction;
    const primaryText = new models.LinkableText(new models.StyledText(didYouMeanMessage, "text/x-apple-as3-nqml"), didYouMeanLinks);
    return new models.SearchResultsMessage(primaryText, null, suggestedSearchAction);
}
// endregion
// region Search Action Builders
/**
 * Build a SearchAction that is for:
 * - Applying a suggested spell correction.
 * - Undoing an automatically applied spell correction.
 *
 * Exported for testing
 *
 * @param suggestedOrUncorrectedTerm The suggestion or uncorrected term to search for.
 * @param resultTerm The original result term.
 * @param spellCorrectionActionType The type of spell correction search.
 */
export function searchActionForSpellCorrection(objectGraph, suggestedOrUncorrectedTerm, resultsTerm, spellCorrectionActionType) {
    // SearchAction for `term` again, disabling spell correction to `correctedTerm`.
    const suggestedSearchAction = new models.SearchAction(suggestedOrUncorrectedTerm, suggestedOrUncorrectedTerm, null, spellCorrectionActionType);
    suggestedSearchAction.spellCheckEnabled = false; // Don't trigger corrections / suggestions again.
    suggestedSearchAction.excludedTerms = [resultsTerm];
    suggestedSearchAction.originatingTerm = resultsTerm;
    return suggestedSearchAction;
}
// endregion
//# sourceMappingURL=search-spell-correction.js.map