/**
 * Manages storing Cohort IDs for Search Landing Page (SLP)
 *
 * # What is Cohort ID?
 * Cohort IDs (a.k.a. cluster IDs) are used to bucket users into different categories, e.g. a gamer.
 * This ID can be used to specify a SLP that is suited to that cateogry of users, e.g. a page featuring more games.
 *
 * # SLP Endpoint Constraints
 * Today, SLPs are not personalized, and rely heavily on CDN caching.
 * We cannot fire a single request to SLP endpoint to have it adapt to a user's cohort based on cookies, etc.
 *
 * As a workaround we:
 * 1. Store the cohort ID for user if we ever load a personalized endpoint, e.g. Today.
 * 2. Send stored cohort ID as a query param on the SLP endpoint, if we have any.
 */
"use strict";
import * as serverData from "../../../foundation/json-parsing/server-data";
// region exports
/**
 * Should be called whenever we recieve a MAPI response from a personalized endpoint.
 * Persists the cohort id that may be present in given MAPI respose for specified user.
 * @param dsid The user's dsid.
 * @param metaDataProviding MAPI response that may contain cohort id for current user.
 */
export function storeCohortIdForUserFromResponse(objectGraph, dsid, metaDataProviding) {
    if (serverData.isNullOrEmpty(dsid)) {
        return;
    }
    const cohortId = cohortIdFromResponse(metaDataProviding);
    if (serverData.isNullOrEmpty(cohortId)) {
        return;
    }
    setCohortIdForDSID(objectGraph, dsid, cohortId);
}
/**
 * Return the stored cohort id for given dsid.
 * @param dsid The DSID of user to fetch cohort id for.
 */
export function cohortIdForUser(objectGraph, dsid) {
    if (serverData.isNullOrEmpty(dsid)) {
        return null;
    }
    return getCohortIdForDSID(objectGraph, dsid);
}
/**
 * Deletes cohort id for given user. For testing.
 */
export function deleteCohortIdForUser(objectGraph, dsid) {
    setCohortIdForDSID(objectGraph, dsid, undefined);
}
// endregion
// region Internals
/**
 * Given a top-level MAPI response `metaDataProviding`, returns the cohort ID, if any.
 * @param metaDataProviding A MAPI Response that may contain a `meta.metrics` blob with `clusterId`
 */
function cohortIdFromResponse(metaDataProviding) {
    return serverData.asString(metaDataProviding, "meta.metrics.clusterId");
}
/**
 * Converts a DSID into a dictionary key for storing cohort ID.
 * @param dsid DSID to generate storage key for.
 */
function cohortIDStorageKeyForDSID(dsid) {
    return dsid + "-cohort-id";
}
/**
 * Set the stored cohort id for given user (by dsid).
 * @param dsid DSID to associate cohortId with
 * @param cohortId The cohort id for user.
 */
function setCohortIdForDSID(objectGraph, dsid, cohortId) {
    const cohortForDSIDKey = cohortIDStorageKeyForDSID(dsid);
    objectGraph.storage.storeString(cohortForDSIDKey, cohortId);
}
/**
 * Gets the stored cohort id for given user by dsid.
 * @param dsid DSID to get cohort id for.
 */
function getCohortIdForDSID(objectGraph, dsid) {
    const cohortForDSIDKey = cohortIDStorageKeyForDSID(dsid);
    return objectGraph.storage.retrieveString(cohortForDSIDKey);
}
//# sourceMappingURL=search-landing-cohort.js.map