/**
 * Builds metrics entities for Guided Search
 */
import { ImpressionMetrics, } from "../../../api/models";
import { isDefinedNonNullNonEmpty, isNullOrEmpty } from "../../../foundation/json-parsing/server-data";
import { createMetricsClickData, createMetricsSearchData } from "../../metrics/builder";
import { createBasicLocation, currentPosition } from "../../metrics/helpers/location";
// region Action Metrics
/**
 * Add click + search metrics data to guided search token action
 * @param objectGraph The dependency graph for the App Store.
 * @param action Action to instrument, e.g. toggle or rewrite.
 * @param targetToken The token display label, which may be toggleable.
 * @param searchTerm The search term for which this token action was returned for.
 * @param metricsOptions The metrics option use for adding instrumentation to token toggle.
 */
export function addEventsToGuidedSearchTokenAction(objectGraph, action, targetToken, searchTerm, metricsOptions) {
    // Click Fields
    const actionType = "guidedSearch";
    const targetType = "guidedLabel";
    const clickFields = {
        actionType: actionType,
        location: createBasicLocation(objectGraph, {
            pageInformation: null,
            locationTracker: metricsOptions.locationTracker,
            targetType: targetType,
        }, targetToken),
        searchTerm: searchTerm,
    };
    // Search Fields
    const searchFields = {
        targetId: targetToken,
    };
    // SSS: Clicks must be before Search
    const clickData = createMetricsClickData(objectGraph, targetToken, targetType, clickFields, ["guidedSearch"]);
    action.actionMetrics.addMetricsData(clickData);
    const searchData = createMetricsSearchData(objectGraph, searchTerm, targetType, actionType, null, searchFields, [
        "guidedSearch",
    ]);
    action.actionMetrics.addMetricsData(searchData);
}
export function addEventsToGuidedSearchTokenEntityChangeAction(objectGraph, action, searchTerm, targetEntity, metricsOptions) {
    // Click Fields
    const actionType = "hint";
    const targetType = "hintsEntity";
    const clickFields = {
        actionType: actionType,
        location: createBasicLocation(objectGraph, {
            pageInformation: null,
            locationTracker: metricsOptions.locationTracker,
            targetType: targetType,
        }, targetEntity),
        searchTerm: searchTerm,
    };
    // Search Fields
    const searchFields = {
        targetId: targetEntity,
    };
    // SSS: Clicks must be before Search
    const clickData = createMetricsClickData(objectGraph, targetEntity, targetType, clickFields);
    action.actionMetrics.addMetricsData(clickData);
    const searchData = createMetricsSearchData(objectGraph, searchTerm, targetType, actionType, null, searchFields);
    action.actionMetrics.addMetricsData(searchData);
}
// endregion
// region Page Metrics
/**
 * Build a `MetricsFields` for guided search fields used for page and impression events.
 * This is attached onto `MetricsPageInformation` for consumption during page and impression field generation.
 */
export function guidedSearchPageInformationFields(objectGraph, request, guidedSearchData) {
    const fields = {};
    // field for what tokens were selected on page.
    if (isDefinedNonNullNonEmpty(request.guidedSearchTokens)) {
        fields["searchSelectedGuidedFacets"] = request.guidedSearchTokens;
    }
    // field for what the server computed final query was
    if (guidedSearchData && guidedSearchData.finalTerm) {
        fields["searchGuidedFinalQuery"] = guidedSearchData.finalTerm;
    }
    if (isNullOrEmpty(fields)) {
        return undefined;
    }
    return fields;
}
// endregion
// region Impression Metrics
export function addImpressionMetricsToGuidedSearchToken(objectGraph, token, type, metricsOptions) {
    const tokenIndex = currentPosition(metricsOptions.locationTracker);
    /**
     * ID for parent is index. This should be explicit but is not in how we do metrics in JS today (sequential IDs are added later)
     */
    const impressionFields = {
        impressionIndex: tokenIndex,
        id: tokenIndex.toString(),
        idType: "sequential",
        name: token.displayName,
        impressionType: type,
        parentId: "search-revisions",
    };
    token.impressionMetrics = new ImpressionMetrics(impressionFields);
}
/**
 * Add imaginary parent container for search results
 * <rdar://problem/70515816> Guided Search: Tech Debt: Trim parent impression tech debt off SearchResults (JS Compatibility Breaking)
 */
export function addSearchResultParentImpressionMetrics(objectGraph, searchResultsPage, metricsOptions) {
    const parentIndex = currentPosition(metricsOptions.locationTracker);
    searchResultsPage.resultsParentImpressionMetrics = new ImpressionMetrics({
        impressionIndex: parentIndex,
        impressionType: "SearchResults",
        idType: "relationship",
        id: "search-results",
        name: "Search Results",
    });
}
/**
 * Add an imaginary parent container for guided search tokens
 * <rdar://problem/70515816> Guided Search: Tech Debt: Trim parent impression tech debt off SearchResults (JS Compatibility Breaking)
 */
export function addGuidedSearchParentImpressionMetrics(objectGraph, searchResultsPage, metricsOptions) {
    const parentIndex = currentPosition(metricsOptions.locationTracker);
    searchResultsPage.guidedSearchTokensParentImpressionMetrics = new ImpressionMetrics({
        impressionIndex: parentIndex,
        impressionType: "SearchRevisions",
        idType: "relationship",
        id: "search-revisions",
        name: "Search Revisions",
    });
}
// endregion
//# sourceMappingURL=guided-search-metrics.js.map