import { unwrapOptional as unwrap } from "@jet/environment/types/optional";
import { GenericPage, Shelf } from "../../api/models";
import { attributeAsBooleanOrFalse, attributeAsString } from "../../foundation/media/attributes";
import { metricsFromMediaApiObject, dataCollectionFromDataContainer, dataFromDataContainer, } from "../../foundation/media/data-structure";
import { relationship } from "../../foundation/media/relationships";
import { addMetricsEventsToPageWithInformation, metricsPageInformationFromMediaApiResponse, } from "../metrics/helpers/page";
import { addImpressionFields } from "../metrics/helpers/impressions";
import { pushContentLocation, newLocationTracker } from "../metrics/helpers/location";
import { lockupsFromData } from "../lockups/lockups";
import { clientIdentifierForEditorialContextInData } from "../lockups/editorial-context";
import { artworkUseCaseFromShelfStyle } from "../content/content";
import { nextDataRange } from "../builders/pagination";
import { platformPrefersLargeTitles, RoomPageToken } from "./room-common";
export const defaultRoomShelfContentType = "mediumLockup";
/**
 * Provides the shelf content type for featured content kinds that require a particular shelf content type that differs
 * from the default.
 *
 * @param fcKind The featured content ID.
 * @returns {ShelfContentType} The content type for the shelf.
 */
export function preferredShelfStyleForFcKind(fcKind) {
    if (fcKind === null || fcKind === undefined) {
        return null;
    }
    switch (fcKind) {
        case 431 /* FeaturedContentID.AppStore_iAPShelf */:
            return "inAppPurchaseTiledLockup";
        default:
            return null;
    }
}
export function renderRoomPage(objectGraph, data) {
    const roomData = unwrap(dataFromDataContainer(objectGraph, data));
    const roomContents = unwrap(relationship(roomData, "contents"));
    const pageInformation = metricsPageInformationFromMediaApiResponse(objectGraph, "Room", roomData.id, data);
    // Create the token
    const token = new RoomPageToken();
    token.remainingContent = roomContents.data;
    token.shouldFilter = !attributeAsBooleanOrFalse(roomData, "doNotFilter");
    token.metricsPageInformation = pageInformation;
    token.metricsLocationTracker = newLocationTracker();
    const fcKindString = attributeAsString(roomData, "editorialElementKind");
    token.preferredShelfContentType = preferredShelfStyleForFcKind(Number(fcKindString));
    token.clientIdentifierOverride = clientIdentifierForEditorialContextInData(objectGraph, roomData);
    // Render the room
    token.title = unwrap(attributeAsString(roomData, "title"));
    const page = roomPageWithContent(objectGraph, roomContents, token);
    page.title = token.title;
    if (platformPrefersLargeTitles(objectGraph)) {
        page.presentationOptions = ["prefersLargeTitle"];
    }
    addMetricsEventsToPageWithInformation(objectGraph, page, pageInformation);
    return page;
}
/**
 * Shared rendering logic for all kinds of "room" pages
 */
export function roomPageWithContent(objectGraph, roomContents, token, shouldUpdateRemainingContent = true) {
    var _a;
    const shelfStyle = token.preferredShelfContentType || defaultRoomShelfContentType;
    const shelf = new Shelf(shelfStyle);
    const shelfMetricsOptions = {
        id: null,
        kind: null,
        softwareType: null,
        targetType: "swoosh",
        title: token.title,
        pageInformation: token.metricsPageInformation,
        locationTracker: token.metricsLocationTracker,
        idType: "its_contentId",
        recoMetricsData: (_a = metricsFromMediaApiObject(roomContents)) !== null && _a !== void 0 ? _a : {},
    };
    /// add impression fields
    addImpressionFields(objectGraph, shelf, shelfMetricsOptions);
    pushContentLocation(objectGraph, shelfMetricsOptions, token.title);
    shelf.isHorizontal = false;
    shelf.shouldFilterApps = token.shouldFilter;
    const contents = dataCollectionFromDataContainer(roomContents);
    if (shouldUpdateRemainingContent) {
        token.remainingContent = [];
    }
    shelf.items = lockupsFromData(objectGraph, contents, {
        contentUnavailable: (index, _dataItem) => {
            if (shouldUpdateRemainingContent) {
                token.remainingContent = nextDataRange(objectGraph, contents, index);
            }
            return true;
        },
        lockupOptions: {
            metricsOptions: {
                pageInformation: token.metricsPageInformation,
                locationTracker: token.metricsLocationTracker,
            },
            clientIdentifierOverride: token.clientIdentifierOverride,
            artworkUseCase: artworkUseCaseFromShelfStyle(objectGraph, shelfStyle),
        },
    });
    const page = new GenericPage([shelf]);
    if (token.remainingContent.length) {
        page.nextPage = token;
    }
    return page;
}
//# sourceMappingURL=room-page.js.map