import * as validation from "@jet/environment/json/validation";
import { isNothing } from "@jet/environment/types/optional";
import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import * as contentAttributes from "../../content/attributes";
import { supportsVisionOSCompatibleIOSBinaryFromData } from "../../content/content";
/**
 * Create an `EditorsChoice` model from the data.
 *
 * @param objectGraph
 * @param data
 * @returns A model containing editors choice information.
 */
export function createEditorsChoiceModel(objectGraph, data) {
    return validation.context("editorsChoiceItem", () => {
        const supportsVisionOSCompatibleIOSBinary = supportsVisionOSCompatibleIOSBinaryFromData(objectGraph, data);
        if (supportsVisionOSCompatibleIOSBinary) {
            return null;
        }
        const editorialNotes = contentAttributes.contentAttributeAsDictionary(objectGraph, data, "editorialNotes");
        const editorsNotes = serverData.asString(editorialNotes, "standard");
        if (editorsNotes) {
            const editorsChoice = new models.EditorsChoice(editorsNotes);
            const editorialBadgeInfo = contentAttributes.contentAttributeAsDictionary(objectGraph, data, "editorialBadgeInfo");
            if (editorialBadgeInfo) {
                const badgeType = serverData.asString(editorialBadgeInfo, "editorialBadgeType");
                editorsChoice.showsBadge = badgeType && badgeType === "editorialPriority";
            }
            if (!editorsChoice.showsBadge) {
                editorsChoice.title = objectGraph.loc.string("APP_STORE_EDITORS_NOTES");
            }
            return editorsChoice;
        }
        return null;
    });
}
/**
 * Create an `editorsChoice` shelf using the data provided.
 *
 * @param objectGraph
 * @param data
 * @returns A shelf for editors choice.
 */
export function createEditorsChoiceShelf(objectGraph, data) {
    const model = createEditorsChoiceModel(objectGraph, data);
    if (isNothing(model)) {
        return null;
    }
    const shelf = new models.Shelf("editorsChoice");
    shelf.items = [model];
    if (model.showsBadge) {
        // Appears as editors choice, with the "Editors' Choice" badge title and background asset
        shelf.background = {
            type: "editorsChoice",
        };
    }
    else {
        // Appears as editors notes, which is less prominent than editors choice.
        shelf.title = objectGraph.loc.string("ProductPage.Section.EditorsNotes.Title");
    }
    return shelf;
}
//# sourceMappingURL=editors-choice-shelf.js.map