import * as models from "../../../../api/models/index";
import * as serverData from "../../../../foundation/json-parsing/server-data";
import { isNothing, isSome } from "@jet/environment";
import * as mediaAttributes from "../../../../foundation/media/attributes";
import * as mediaRelationship from "../../../../foundation/media/relationships";
import { createArtworkForResource } from "../../../content/artwork/artwork";
import * as offerFormatting from "../../../offers/offer-formatting";
import * as offers from "../../../offers/offers";
import { isEmpty } from "../../../../foundation/util/array-util";
import { makeRoutableArticlePageIntent } from "../../../../api/intents/routable-article-page-intent";
import { getLocale } from "../../../locale";
import { makeRoutableArticlePageCanonicalUrl } from "../../../today/routable-article-page-url-utils";
import { getPlatform } from "../../../preview-platform";
export function createAnnotation(objectGraph, data, isFirstPartyHideableApp, isArcadeApp, suppressTitle, shelfMetrics) {
    const topInApps = mediaRelationship.relationshipCollection(data, "top-in-apps");
    if (isNothing(topInApps) || isEmpty(topInApps)) {
        return null;
    }
    const textPairs = [];
    const items_V3 = [];
    // Create IAP items
    for (const inAppData of topInApps) {
        const name = mediaAttributes.attributeAsString(inAppData, "name");
        if (isNothing(name)) {
            continue;
        }
        const offer = offers.offerDataFromData(objectGraph, inAppData);
        if (isNothing(offer)) {
            continue;
        }
        const price = serverData.asString(offer, "priceFormatted");
        if (isNothing(price)) {
            continue;
        }
        const duration = serverData.asString(offer, "recurringSubscriptionPeriod");
        const numberOfPeriods = serverData.asNumber(offer, "numOfPeriods");
        let nameAndDuration = objectGraph.loc.string("InfoList.IAP.Duration");
        if (objectGraph.client.isiOS && isSome(duration)) {
            nameAndDuration = nameAndDuration.replace("{name}", name);
            const subscriptionRecurrence = offerFormatting.subscriptionRecurrenceForServerRecurrence(objectGraph, duration, numberOfPeriods);
            let timeUnit;
            switch (subscriptionRecurrence === null || subscriptionRecurrence === void 0 ? void 0 : subscriptionRecurrence.type) {
                case "D":
                    timeUnit = "days";
                    break;
                case "W":
                    timeUnit = "weeks";
                    break;
                case "M":
                    timeUnit = "months";
                    break;
                case "Y":
                    timeUnit = "years";
                    break;
                default:
                    nameAndDuration = `${name}`;
                    break;
            }
            if (isSome(timeUnit) && isSome(subscriptionRecurrence)) {
                const formattedDuration = objectGraph.loc.formatDuration(subscriptionRecurrence.periodDuration, timeUnit);
                if (isSome(formattedDuration)) {
                    nameAndDuration = nameAndDuration.replace("{duration}", formattedDuration);
                }
            }
        }
        else {
            nameAndDuration = `${name}`;
        }
        textPairs.push([nameAndDuration, price]);
        items_V3.push({
            $kind: "textPair",
            leadingText: nameAndDuration,
            trailingText: price,
        });
    }
    // Learn More link
    const learnMoreTitle = objectGraph.loc.string("ProductPage.Info.InAppPurchases.LearnMore");
    const learnMoreAction = makeLearnMoreAction(objectGraph, learnMoreTitle);
    if (isSome(learnMoreAction)) {
        if (objectGraph.client.isiOS || objectGraph.client.isMac || objectGraph.client.isVision) {
            // The native support for the `small` spacer was introduced in 2025B. Older clients will just ignore this
            // and show the standard size space, which is still preferable over no space at all.
            items_V3.push({
                $kind: "spacer",
                size: "small",
            });
            // On iOS, macOS & visionOS, we want this to appear as a link.
            const linkedSubstrings = {};
            linkedSubstrings[learnMoreTitle] = learnMoreAction;
            items_V3.push({
                $kind: "linkableText",
                linkableText: new models.LinkableText(new models.StyledText(learnMoreTitle), linkedSubstrings),
            });
        }
        else if (objectGraph.client.isWatch || objectGraph.client.isWeb) {
            // On watchOS & web, we want this to appear as a button.
            items_V3.push({
                $kind: "button",
                action: learnMoreAction,
                style: "infer",
            });
        }
        //  We intentionally leave out tvOS
    }
    const title = suppressTitle ? "" : objectGraph.loc.string("InfoList.InAppPurchases");
    const items = [new models.AnnotationItem(undefined, { textPairs: textPairs })];
    const annotation = new models.Annotation(title, items, objectGraph.loc.string("YES"));
    annotation.items_V3 = items_V3;
    if (objectGraph.client.isVision || preprocessor.GAMES_TARGET) {
        annotation.leadingArtwork = createArtworkForResource(objectGraph, "resource://in.app.purchase");
    }
    return annotation;
}
function makeLearnMoreAction(objectGraph, title) {
    const iapLearnMoreEditorialItemID = objectGraph.bag.inAppPurchasesLearnMoreEditorialItemId;
    if (isNothing(iapLearnMoreEditorialItemID) || isEmpty(iapLearnMoreEditorialItemID)) {
        return null;
    }
    const flowAction = new models.FlowAction("article");
    flowAction.title = title;
    flowAction.pageUrl = `https://apps.apple.com/story/id${objectGraph.bag.inAppPurchasesLearnMoreEditorialItemId}`;
    if (objectGraph.client.isVision) {
        const dismissAction = new models.FlowBackAction("sheetDismiss");
        const compoundAction = new models.CompoundAction([dismissAction, flowAction]);
        compoundAction.title = title;
        return compoundAction;
    }
    else if (objectGraph.client.isWeb) {
        const routableArticlePageIntent = makeRoutableArticlePageIntent({
            ...getLocale(objectGraph),
            ...getPlatform(objectGraph),
            id: iapLearnMoreEditorialItemID,
        });
        flowAction.pageUrl = makeRoutableArticlePageCanonicalUrl(objectGraph, routableArticlePageIntent);
        flowAction.destination = routableArticlePageIntent;
        return flowAction;
    }
    else {
        return flowAction;
    }
}
//# sourceMappingURL=top-in-app-purchases-annotation.js.map