import { isNothing, isSome } from "@jet/environment";
import * as models from "../../../../api/models/index";
import * as serverData from "../../../../foundation/json-parsing/server-data";
import { createArtworkForResource } from "../../../content/artwork/artwork";
import * as contentAttributes from "../../../content/attributes";
export function createAnnotation(objectGraph, data, isFirstPartyHideableApp, isArcadeApp, suppressTitle, shelfMetrics) {
    if (isArcadeApp) {
        return null;
    }
    const sellerData = contentAttributes.contentAttributeAsString(objectGraph, data, "seller");
    const sellerInfoData = contentAttributes.contentAttributeAsDictionary(objectGraph, data, "sellerInfo");
    const icpInfo = contentAttributes.contentAttributeAsDictionary(objectGraph, data, "internetContentProviderInfo");
    const enableSellerICP = objectGraph.bag.enableSellerICPAnnotation && objectGraph.client.isiOS;
    if (isNothing(sellerData)) {
        return null;
    }
    const sellerLabel = contentAttributes.contentAttributeAsString(objectGraph, data, "sellerLabel") ||
        objectGraph.loc.string("Seller");
    let annotation;
    let annotationItems = [];
    const sellerDataName = serverData.asString(sellerInfoData, "name");
    const sellerName = isSome(sellerDataName) && (sellerDataName === null || sellerDataName === void 0 ? void 0 : sellerDataName.length) > 0 ? sellerDataName : sellerData;
    const title = suppressTitle ? "" : sellerLabel;
    let annotationSummary = null;
    if (isSome(sellerInfoData)) {
        const [newItems, summary] = createSellerInfoAnnotationItemsAndSummary(objectGraph, sellerInfoData);
        annotationSummary = summary;
        annotationItems = annotationItems.concat(newItems);
    }
    if (enableSellerICP && isSome(icpInfo)) {
        const annotationItem = setupAnnotationForKey(objectGraph, "filingNumber", objectGraph.loc.string("ProductPage.Info.ICPNumber.Title"), icpInfo);
        // If no other seller info, include seller title in ICP annotation
        if (annotationItems.length === 0) {
            const annotationTitle = new models.AnnotationItem(sellerName);
            annotationItems.push(annotationTitle);
        }
        if (isSome(annotationItem)) {
            annotationItems.push(annotationItem);
        }
    }
    if (annotationItems.length > 0) {
        if (isNothing(annotationSummary)) {
            annotationSummary = annotationItems.length > 1 ? sellerName : undefined;
        }
        annotation = new models.Annotation(title, annotationItems, annotationSummary, undefined);
    }
    else {
        const sellerItem = new models.AnnotationItem(sellerData);
        annotation = new models.Annotation(title, [sellerItem]);
    }
    if (isSome(annotation) && (objectGraph.client.isVision || preprocessor.GAMES_TARGET)) {
        annotation.leadingArtwork = createArtworkForResource(objectGraph, "resource://person.crop.app");
    }
    return annotation;
}
/**
 * Create the seller info annotation items on the product page.
 *
 * @param sellerData The sellerInfo map from the product page JSON fetch response.
 * @returns A tuple consisting of a list of seller info annotation items and an optional string for the summary, if necessary to provide.
 */
function createSellerInfoAnnotationItemsAndSummary(objectGraph, sellerData) {
    const sellerInfoAnnotationItems = [];
    const locKeysMap = {
        name: "ProductPage.SellerInfo.Name.Title",
        dunsNumber: "ProductPage.SellerInfo.DunsNumber.Title",
        address: "ProductPage.SellerInfo.Address.Title",
        brn: "ProductPage.SellerInfo.BusinessRegistrationNumber.Title",
        phoneNumber: "ProductPage.SellerInfo.PhoneNumber.Title",
        email: "ProductPage.SellerInfo.Email.Title",
        usci: "ProductPage.SellerInfo.UnifiedSocialCreditIdentifier.Title",
    };
    const sellerDataName = serverData.asString(sellerData, "name");
    const sellerName = isSome(sellerDataName) && (sellerDataName === null || sellerDataName === void 0 ? void 0 : sellerDataName.length) > 0 ? sellerDataName : objectGraph.loc.string("Seller");
    const traderBooleanProvided = isSome(serverData.asString(sellerData, "isTrader"));
    if (traderBooleanProvided) {
        const hasTraderInfo = serverData.asBoolean(sellerData, "isTrader");
        // Trader Identified
        const traderVerificationString = hasTraderInfo
            ? "ProductPage.SellerInfo.VerifiedSeller.Subtitle"
            : "ProductPage.SellerInfo.UnverifiedSeller.Subtitle";
        sellerInfoAnnotationItems.push(new models.AnnotationItem(objectGraph.loc.string(traderVerificationString).replace("{seller}", sellerName)));
        if (!hasTraderInfo) {
            return [sellerInfoAnnotationItems, sellerName];
        }
    }
    // List remaining annotations from seller info
    // https://developer.apple.com/help/app-store-connect/manage-compliance-information/manage-european-union-digital-services-act-trader-requirements/
    let summary = null;
    // If there's only one key in seller data and it's not the seller name, put "See Details" summary.
    if (Object.keys(sellerData).length === 1 && !Object.keys(sellerData).includes("name")) {
        summary = objectGraph.loc.string("ProductPage.SellerInfo.SeeDetails.Title");
    }
    for (const [key, val] of Object.entries(locKeysMap)) {
        // Don't need a redundant name field if already included in DSA trader verification subtitle.
        if (key === "name" && traderBooleanProvided) {
            continue;
        }
        const annotation = setupAnnotationForKey(objectGraph, key, objectGraph.loc.string(val), sellerData);
        if (isSome(annotation)) {
            sellerInfoAnnotationItems.push(annotation);
        }
    }
    return [sellerInfoAnnotationItems, summary];
}
function setupAnnotationForKey(objectGraph, key, title, sellerData) {
    let itemText = serverData.asString(sellerData, key);
    if (isSome(itemText)) {
        if (key === "brn") {
            itemText =
                itemText + "\n" + objectGraph.loc.string("ProductPage.SellerInfo.BusinessRegistrationNumber.Subtitle");
        }
        return new models.AnnotationItem(itemText, { heading: title });
    }
    return null;
}
//# sourceMappingURL=seller-annotation.js.map