import * as models from "../../../../api/models/index";
import { asString, attributeAsDictionary } from "@apple-media-services/media-api";
import { isSome } from "@jet/environment";
import { isNotEmpty } from "../../../../foundation/util/array-util";
import { createArtworkForResource } from "../../../content/artwork/artwork";
/**
 * Create an annotation for a game license, currently supported for the Vietnamese storefront.
 */
export function createAnnotation(objectGraph, data, isFirstPartyHideableApp, isArcadeApp, suppressTitle, shelfMetrics) {
    const gameLicense = attributeAsDictionary(data, "licenses.gameLicenseVietnam");
    const items = [plainText(objectGraph.loc.string("InfoList.GameLicense.Disclaimer"))];
    // Add license id
    const gameLicenseId = asString(gameLicense, "licenseId");
    if (isSome(gameLicenseId) && isNotEmpty(gameLicenseId)) {
        items.push(spacer(), heading(objectGraph.loc.string("InfoList.GameLicense.Id.Title")), plainText(gameLicenseId));
    }
    else {
        // The license id is required. If this is missing, we should not display this annotation.
        return null;
    }
    // Add license url
    const gameLicenseUrl = asString(gameLicense, "licenseUrl");
    if (isSome(gameLicenseUrl) && isNotEmpty(gameLicenseUrl)) {
        if (objectGraph.client.isWatch) {
            items.push(buttonLink(objectGraph.loc.string("InfoList.GameLicense.Url.ButtonTitle"), gameLicenseUrl));
        }
        else {
            items.push(spacer(), heading(objectGraph.loc.string("InfoList.GameLicense.Url.Title")), externalLink(gameLicenseUrl));
        }
    }
    const title = suppressTitle ? "" : objectGraph.loc.string("InfoList.GameLicense.Title");
    const annotation = new models.Annotation(title, [], gameLicenseId);
    annotation.items_V3 = items;
    if (objectGraph.client.isVision) {
        annotation.leadingArtwork = createArtworkForResource(objectGraph, "systemimage://text.page");
    }
    return annotation;
}
// region Annotation Helpers
function spacer() {
    return {
        $kind: "spacer",
    };
}
function heading(text) {
    return {
        $kind: "linkableText",
        linkableText: new models.LinkableText(new models.StyledText(`**${text}**`)),
    };
}
function plainText(text) {
    return {
        $kind: "linkableText",
        linkableText: new models.LinkableText(new models.StyledText(text)),
    };
}
function externalLink(url) {
    const linkedSubstrings = {};
    linkedSubstrings[url] = new models.ExternalUrlAction(url, false);
    return {
        $kind: "linkableText",
        linkableText: new models.LinkableText(new models.StyledText(url), linkedSubstrings),
    };
}
function buttonLink(title, url) {
    const action = new models.ExternalUrlAction(url, false);
    action.title = title;
    return {
        $kind: "button",
        action: action,
    };
}
//# sourceMappingURL=game-license-annotation.js.map