import * as validation from "@jet/environment/json/validation";
import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../foundation/media/attributes";
import * as mediaDataStructure from "../../../foundation/media/data-structure";
import * as mediaRelationship from "../../../foundation/media/relationships";
import * as contentAttributes from "../../content/attributes";
import * as content from "../../content/content";
import * as productCapabilities from "../../content/product-capabilities";
import * as metricsHelpersClicks from "../../metrics/helpers/clicks";
import * as metricsHelpersPage from "../../metrics/helpers/page";
import * as reviews from "../reviews";
import * as informationShelf from "./annotations/annotations";
import * as versionHistoryShelves from "./version-history-shelves";
import * as accessibilityShelves from "../shelves/accessibility-shelves";
import { isSome } from "@jet/environment";
/**
 * Create a shelf of action links for the product page.
 *
 * @param data The raw data response for a product page JSON fetch.
 * @param shelfMetrics The product page shelf metrics.
 * @param shelfContext A collection of any other variables used when creating this shelf.
 * @returns A shelf of action links.
 */
export function create(objectGraph, data, shelfMetrics, shelfContext) {
    return validation.context("createActionLinks", () => {
        if (serverData.isNullOrEmpty(data)) {
            return null;
        }
        if (objectGraph.client.isWatch) {
            return standardActionLinks(objectGraph, data, shelfContext.productTitle, shelfMetrics, shelfContext.shouldShowRatingsAndReviews, shelfContext.isFirstPartyHideableApp, shelfContext.isBundle, shelfContext.isPreorder, shelfContext.isArcadeApp, shelfContext.isFreeProduct, shelfContext.deviceFamily);
        }
        return null;
    });
}
function clickOptionsForActionLinkAction(objectGraph, productId, shelfMetrics) {
    return {
        targetType: "button",
        actionType: "navigate",
        id: productId,
        pageInformation: shelfMetrics.metricsPageInformation,
        locationTracker: shelfMetrics.locationTracker,
    };
}
function standardActionLinks(objectGraph, productData, productTitle, shelfMetrics, shouldShowRatingsAndReviews, isFirstPartyHideableApp, isBundle, isPreorder, isArcadeApp, isFreeProduct, deviceFamily) {
    return validation.context("actionLinkShelf", () => {
        const shelf = new models.Shelf("action");
        const items = [];
        // Ratings & Reviews
        if (shouldShowRatingsAndReviews) {
            const reviewsData = mediaRelationship.relationship(productData, "reviews");
            const ratingsData = mediaAttributes.attributeAsDictionary(productData, "userRating");
            ratingsData.ratingAverage = serverData.asNumber(ratingsData, "value");
            ratingsData.adamId = productData.id;
            ratingsData.isBundle = isBundle;
            ratingsData.supportUrl = contentAttributes.contentAttributeAsString(objectGraph, productData, "supportURLForLanguage");
            const reviewItems = mediaDataStructure.dataCollectionFromDataContainer(reviewsData);
            const appIcon = content.iconFromData(objectGraph, productData, {
                useCase: 2 /* content.ArtworkUseCase.LockupIconMedium */,
            });
            const ratingsAndReviewsLink = reviews.reviewsPageActionFromReviewsData(objectGraph, objectGraph.client.guid, ratingsData, reviewItems, reviewsData.next, productTitle, appIcon, false, false);
            ratingsAndReviewsLink.title = objectGraph.loc.string("ProductPage.Section.Reviews.Title");
            items.push(ratingsAndReviewsLink);
        }
        // Version History
        const versions = contentAttributes.contentAttributeAsArrayOrEmpty(objectGraph, productData, "versionHistory");
        if (versions.length > 0 && !isFirstPartyHideableApp && !isBundle && !isPreorder) {
            const versionHistoryLinkAction = versionHistoryShelves.versionHistoryPageAction(objectGraph, versions, productData.id);
            // Metrics (page information configured in version history builder)
            metricsHelpersClicks.addClickEventToAction(objectGraph, versionHistoryLinkAction, clickOptionsForActionLinkAction(objectGraph, productData.id, shelfMetrics));
            items.push(versionHistoryLinkAction);
        }
        // Accessibility
        if (isSome(deviceFamily) &&
            accessibilityShelves.shouldShowAccessibilitySection(objectGraph, productData, deviceFamily)) {
            const action = accessibilityShelves.createSeeIntermediaryDetailsAction(objectGraph, productData, deviceFamily);
            metricsHelpersClicks.addClickEventToAction(objectGraph, action, clickOptionsForActionLinkAction(objectGraph, productData.id, shelfMetrics));
            items.push(action);
        }
        // Info List
        const infoShelf = informationShelf.create(objectGraph, productData, isFirstPartyHideableApp, shelfMetrics, null, isArcadeApp, isFreeProduct, false);
        const infoPage = new models.GenericPage([infoShelf]);
        const infoLinkAction = new models.FlowAction("page");
        infoLinkAction.title = objectGraph.loc.string("ProductPage.Section.Information.Title");
        infoPage.title = infoLinkAction.title;
        // Metrics
        shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "information");
        const infoPageInformation = metricsHelpersPage.pageInformationForActionLinkPage(objectGraph, productData.id, "info");
        metricsHelpersPage.addMetricsEventsToPageWithInformation(objectGraph, infoPage, infoPageInformation);
        metricsHelpersClicks.addClickEventToAction(objectGraph, infoLinkAction, clickOptionsForActionLinkAction(objectGraph, productData.id, shelfMetrics));
        infoLinkAction.pageData = infoPage;
        items.push(infoLinkAction);
        // Capabilities
        const capabilities = productCapabilities.productCapabilitiesFromData(objectGraph, productData, isFreeProduct);
        if (capabilities.length > 0) {
            const capabilitiesShelf = new models.Shelf("productCapability");
            capabilitiesShelf.items = capabilities;
            const capabilitiesPage = new models.GenericPage([capabilitiesShelf]);
            const capabilitiesLinkAction = new models.FlowAction("page");
            capabilitiesLinkAction.title = objectGraph.loc.string("ProductPage.Section.Supports.Title");
            capabilitiesPage.title = capabilitiesLinkAction.title;
            // Metrics
            shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "supports");
            const capabilitiesPageInformation = metricsHelpersPage.pageInformationForActionLinkPage(objectGraph, productData.id, "supports");
            metricsHelpersPage.addMetricsEventsToPageWithInformation(objectGraph, capabilitiesPage, capabilitiesPageInformation);
            metricsHelpersClicks.addClickEventToAction(objectGraph, capabilitiesLinkAction, clickOptionsForActionLinkAction(objectGraph, productData.id, shelfMetrics));
            capabilitiesLinkAction.pageData = capabilitiesPage;
            items.push(capabilitiesLinkAction);
        }
        // Privacy Link
        const privacyPolicyUrl = contentAttributes.contentAttributeAsString(objectGraph, productData, "privacyPolicyUrl");
        if (privacyPolicyUrl) {
            const privacyLinkAction = new models.ExternalUrlAction(privacyPolicyUrl, false);
            // Metrics
            metricsHelpersClicks.addClickEventToAction(objectGraph, privacyLinkAction, clickOptionsForActionLinkAction(objectGraph, productData.id, shelfMetrics));
            privacyLinkAction.title = objectGraph.loc.string("PRIVACY_POLICY");
            items.push(privacyLinkAction);
        }
        const traderData = contentAttributes.contentAttributeAsDictionary(objectGraph, productData, "sellerInfo");
        const safetyAndComplianceURL = serverData.asString(traderData, "safetyAndComplianceUrl");
        if ((safetyAndComplianceURL === null || safetyAndComplianceURL === void 0 ? void 0 : safetyAndComplianceURL.length) > 0) {
            const safetyLinkAction = new models.ExternalUrlAction(safetyAndComplianceURL, false);
            safetyLinkAction.title = objectGraph.loc.string("ProductPage.Section.SafetyAndCompliance.Title");
            items.push(safetyLinkAction);
        }
        shelf.items = items;
        shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "actionLinks");
        if (items.length > 0) {
            return shelf;
        }
        else {
            return null;
        }
    });
}
//# sourceMappingURL=action-links-shelf.js.map