//
//  product-page-shelf-ordering.ts
//  AppStoreKit
//
//  Created by Dersu Abolfathi on 2/15/17.
//  Copyright (c) 2017 Apple Inc. All rights reserved.
//
import { ProductPageSectionMapping } from "../product-page-section-mapping";
import * as iOSOrdering from "./product-page-shelf-ordering-ios";
import * as iOSArcadeOrdering from "./product-page-shelf-ordering-ios-arcade";
import * as macOrdering from "./product-page-shelf-ordering-mac";
import * as macArcadeOrdering from "./product-page-shelf-ordering-mac-arcade";
import * as tvOSOrdering from "./product-page-shelf-ordering-tvos";
import * as tvOSArcadeOrdering from "./product-page-shelf-ordering-tvos-arcade";
import * as watchOSOrdering from "./product-page-shelf-ordering-watch";
import * as visionOSOrdering from "./product-page-shelf-ordering-visionos";
import * as visionOSArcadeOrdering from "./product-page-shelf-ordering-visionos-arcade";
import * as webOrdering from "./product-page-shelf-ordering-web";
import * as webArcadeOrdering from "./product-page-shelf-ordering-web-arcade";
export function orderingForPlatform(objectGraph, hostPlatform, supportsArcade) {
    switch (hostPlatform) {
        case "tvOS":
            return supportsArcade ? tvOSArcadeOrdering : tvOSOrdering;
        case "watchOS":
            return watchOSOrdering;
        case "iOS":
            return supportsArcade ? iOSArcadeOrdering : iOSOrdering;
        case "macOS":
            return supportsArcade ? macArcadeOrdering : macOrdering;
        case "xrOS":
            return supportsArcade ? visionOSArcadeOrdering : visionOSOrdering;
        case "web":
            return supportsArcade ? webArcadeOrdering : webOrdering;
        default:
            return iOSOrdering;
    }
}
/**
 * Determines the overall section ordering for a product page.
 * @param objectGraph
 * @param isPurchased = Whether the requested ordering is for the purchased state.
 * @param hostPlatform
 * @param supportsArcade
 * @param isCompact - ordering for compact display
 * @param isDownloading Whether the section ordering is for the downloading order of a product page.
 * @param options - The options to use for the product page configuration.
 * note: - This function is backed by a small, straightforward set of human-readable
 * arrays. Customization with regards to 'spotlighting' (placing a section at the
 * top of the page, depending on how the product page was reached) is applied through
 * the 'options' parameter.
 * @return The ordering for the overall product page sections.
 */
export function sectionOrdering(objectGraph, isPurchased, hostPlatform, supportsArcade, isCompact, isDownloading, options) {
    const ordering = orderingForPlatform(objectGraph, hostPlatform, supportsArcade);
    // Create a shallow copy of the appropriate ordering
    let sections = [];
    if (isPurchased) {
        sections =
            isCompact && ordering.purchasedOrderingCompact
                ? ordering.purchasedOrderingCompact.slice()
                : ordering.purchasedOrdering.slice();
    }
    else if (isDownloading && ordering.downloadingOrdering) {
        sections =
            isCompact && ordering.downloadingOrderingCompact
                ? ordering.downloadingOrderingCompact.slice()
                : ordering.downloadingOrdering.slice();
    }
    else {
        sections =
            isCompact && ordering.notPurchasedOrderingCompact
                ? ordering.notPurchasedOrderingCompact.slice()
                : ordering.notPurchasedOrdering.slice();
    }
    const spotlightIndex = isPurchased ? ordering.purchasedSpotlightIndex : ordering.notPurchasedSpotlightIndex;
    // Case: Our options dictate spotlighting of a section
    if (options && options.spotlightSection) {
        // Re-Insert the section at the spotlight index
        const index = indexOfSectionInOrdering(options.spotlightSection, sections);
        if (index !== -1) {
            sections.splice(index, 1);
            sections.splice(spotlightIndex, 0, options.spotlightSection);
        }
    }
    return sections;
}
/**
 * Determines the index of a particular section within an ordering of sections.
 * @param section     The section in question.
 * @param ordering    The ordering in which the section's index is desired.
 * @returns The index of the section, or -1 if it does not appear in the ordering.
 */
function indexOfSectionInOrdering(section, ordering) {
    for (let index = 0; index < ordering.length; index++) {
        const aSection = ordering[index];
        if (section.sectionType === aSection.sectionType &&
            section.shelfId === aSection.shelfId &&
            JSON.stringify(section.shelfBasedMappedIds) === JSON.stringify(aSection.shelfBasedMappedIds)) {
            return index;
        }
    }
    return -1;
}
/**
 * Determines the appropriate product page section for the iAP type.
 * @param isSubscription The type for the iAP product.
 * @returns {ProductPageSection} The section appropriate for this iAP type.
 */
export function inAppPurchaseSection(isSubscription) {
    if (isSubscription) {
        return new ProductPageSectionMapping("shelf", "subscriptions");
    }
    else {
        return new ProductPageSectionMapping("shelf", "inAppPurchases");
    }
}
//# sourceMappingURL=product-page-shelf-ordering.js.map