//
//  product-page.ts
//  AppStoreKit
//
//  Created by Kevin MacWhinnie on 8/17/16.
//  Copyright (c) 2016 Apple Inc. All rights reserved.
//
import * as validation from "@jet/environment/json/validation";
import * as models from "../../api/models";
import * as serverData from "../../foundation/json-parsing/server-data";
import * as productPageUtil from "./product-page-util";
import { sectionTitleForPlatform } from "./reviews";
import { legacyProductPageSectionsFromMappings } from "./shelf-based/product-page-section-mapping";
import * as productOrdering from "./shelf-based/product-page-shelf-ordering/product-page-shelf-ordering";
import * as productMediaShelf from "./shelves/product-media-shelf";
/**
 * Creates a non-ShelfBasedProductPage from a ShelfBased on
 * @param objectGraph
 * @param shelfBasedProductPageModel
 * @param isArcadeApp
 * @param options
 */
export function createProductPageFromShelfBasedProductPage(objectGraph, shelfBasedProductPageModel, isArcadeApp, options) {
    return validation.context("createProductPageFromShelfBasedProductPage", () => {
        var _a, _b, _c, _d;
        const productPage = new models.ProductPage();
        // Lockup Level Properties
        productPage.adamId = shelfBasedProductPageModel.lockup.adamId;
        productPage.bundleId = shelfBasedProductPageModel.lockup.bundleId;
        productPage.icon = shelfBasedProductPageModel.lockup.icon;
        productPage.isEditorsChoice = shelfBasedProductPageModel.lockup.isEditorsChoice;
        productPage.ordinal = shelfBasedProductPageModel.lockup.ordinal;
        productPage.title = shelfBasedProductPageModel.lockup.title;
        productPage.subtitle = shelfBasedProductPageModel.lockup.subtitle;
        productPage.developerTagline = shelfBasedProductPageModel.lockup.developerTagline;
        productPage.editorialTagline = shelfBasedProductPageModel.lockup.editorialTagline;
        productPage.editorialDescription = shelfBasedProductPageModel.lockup.editorialDescription;
        productPage.shortEditorialDescription = shelfBasedProductPageModel.lockup.shortEditorialDescription;
        productPage.rating = shelfBasedProductPageModel.lockup.rating;
        productPage.ratingCount = shelfBasedProductPageModel.lockup.ratingCount;
        productPage.ageRating = shelfBasedProductPageModel.lockup.ageRating;
        productPage.buttonAction = shelfBasedProductPageModel.lockup.buttonAction;
        productPage.offerDisplayProperties = shelfBasedProductPageModel.lockup.offerDisplayProperties;
        productPage.titleOfferDisplayProperties = shelfBasedProductPageModel.titleOfferDisplayProperties;
        productPage.clickAction = shelfBasedProductPageModel.lockup.clickAction;
        productPage.children = shelfBasedProductPageModel.lockup.children;
        productPage.contextMenuData = shelfBasedProductPageModel.lockup.contextMenuData;
        productPage.shelfBackground = shelfBasedProductPageModel.lockup.shelfBackground;
        productPage.searchAd = shelfBasedProductPageModel.lockup.searchAd;
        productPage.searchAdOpportunity = shelfBasedProductPageModel.lockup.searchAdOpportunity;
        productPage.crossLinkTitle = shelfBasedProductPageModel.lockup.crossLinkTitle;
        productPage.crossLinkSubtitle = shelfBasedProductPageModel.lockup.crossLinkSubtitle;
        productPage.tertiaryTitle = shelfBasedProductPageModel.lockup.tertiaryTitle;
        productPage.tertiaryTitleAction = shelfBasedProductPageModel.lockup.tertiaryTitleAction;
        productPage.tertiaryTitleArtwork = shelfBasedProductPageModel.lockup.tertiaryTitleArtwork;
        productPage.flowPreviewActionsConfiguration = shelfBasedProductPageModel.lockup.flowPreviewActionsConfiguration;
        productPage.productDescription = shelfBasedProductPageModel.lockup.productDescription;
        productPage.itemBackground = shelfBasedProductPageModel.lockup.itemBackground;
        // Page Level Properties
        productPage.developerAction = shelfBasedProductPageModel.developerAction;
        productPage.ageRatingAction = shelfBasedProductPageModel.ageRatingAction;
        productPage.logoArtwork = shelfBasedProductPageModel.logoArtwork;
        productPage.navigationBarIconArtwork = shelfBasedProductPageModel.navigationBarIconArtwork;
        productPage.uberArtwork = (_a = shelfBasedProductPageModel.uber) === null || _a === void 0 ? void 0 : _a.artwork;
        productPage.uberArtworkForCompactDisplay = (_b = shelfBasedProductPageModel.uber) === null || _b === void 0 ? void 0 : _b.compactArtwork;
        productPage.uberVideo = (_c = shelfBasedProductPageModel.uber) === null || _c === void 0 ? void 0 : _c.video;
        productPage.uberStyle = (_d = shelfBasedProductPageModel.uber) === null || _d === void 0 ? void 0 : _d.style;
        productPage.media = allProductMediaFromShelfBasedProductPage(shelfBasedProductPageModel);
        if (serverData.isDefinedNonNull(shelfBasedProductPageModel.shelfMapping["informationRibbon"])) {
            productPage.badges = productPageUtil.badgesFromInfoRibbonShelf(objectGraph, shelfBasedProductPageModel.shelfMapping["informationRibbon"]);
        }
        else if (objectGraph.host.isWatch || objectGraph.host.isTV) {
            // copy sidepack badge for watch and tv.
            productPage.badges = shelfBasedProductPageModel.badges;
        }
        productPage.shareAction = shelfBasedProductPageModel.shareAction;
        productPage.pageMetrics = shelfBasedProductPageModel.pageMetrics;
        productPage.pageRenderMetrics = shelfBasedProductPageModel.pageRenderMetrics;
        productPage.isComplete = !shelfBasedProductPageModel.isIncomplete;
        productPage.hasDarkUserInterfaceStyle = shelfBasedProductPageModel.hasDarkUserInterfaceStyle;
        productPage.mediaSectionTitle = objectGraph.client.isTV
            ? null
            : objectGraph.loc.string("ProductPage.Section.ScreenshotsPreview.Title");
        productPage.expandedOfferDetails = shelfBasedProductPageModel.expandedOfferDetails;
        productPage.regularPriceFormatted = shelfBasedProductPageModel.regularPriceFormatted;
        productPage.theme = shelfBasedProductPageModel.theme;
        productPage.externalVersionIdentifier = shelfBasedProductPageModel.externalVersionIdentifier;
        productPage.updateBuyParams = shelfBasedProductPageModel.updateBuyParams;
        productPage.appPlatforms = shelfBasedProductPageModel.appPlatforms;
        productPage.descriptionHeader = shelfBasedProductPageModel.descriptionHeader;
        productPage.description = shelfBasedProductPageModel.description;
        productPage.banner = shelfBasedProductPageModel.banner;
        productPage.secondaryBanner = shelfBasedProductPageModel.secondaryBanner;
        productPage.fullProductFetchedAction = shelfBasedProductPageModel.fullProductFetchedAction;
        productPage.appPromotionDetailPageFlowAction = shelfBasedProductPageModel.appPromotionDetailPageFlowAction;
        productPage.pageRefreshPolicy = shelfBasedProductPageModel.pageRefreshPolicy;
        productPage.purchasedOrdering = legacyProductPageSectionsFromMappings(objectGraph, productOrdering.sectionOrdering(objectGraph, true, objectGraph.host.platform, isArcadeApp, false, false, options));
        productPage.notPurchasedOrdering = legacyProductPageSectionsFromMappings(objectGraph, productOrdering.sectionOrdering(objectGraph, false, objectGraph.host.platform, isArcadeApp, false, false, options));
        productPage.shelfMapping = productShelfMappingFromShelfBasedProductPage(objectGraph, shelfBasedProductPageModel);
        productPage.alwaysAllowReviews = shelfBasedProductPageModel.alwaysAllowReviews;
        return productPage;
    });
}
function productShelfMappingFromShelfBasedProductPage(objectGraph, shelfBasedPage) {
    const shelfMapping = {};
    for (const knownShelfId of models.legacyProductPageKnownShelfIds) {
        let shelf;
        if (knownShelfId === "reviews") {
            shelf = productReviewsShelfFromShelfBasedProductPage(objectGraph, shelfBasedPage);
        }
        else {
            shelf = shelfBasedPage.shelfMapping[knownShelfId];
        }
        if (serverData.isDefinedNonNullNonEmpty(shelf)) {
            shelfMapping[knownShelfId] = shelf;
        }
    }
    return shelfMapping;
}
export function allProductMediaFromShelfBasedProductPage(shelfBasedPage) {
    const productMedia = [];
    for (const shelf of Object.values(shelfBasedPage.shelfMapping)) {
        if (productMediaShelf.isProductMediaShelf(shelf)) {
            const productMediaMetaData = shelf.contentsMetadata;
            const productMediaItems = shelf.items;
            productMedia.push(new models.ProductMedia(productMediaItems, productMediaMetaData.platform, productMediaMetaData.allPlatforms, productMediaMetaData.platformDescription, productMediaMetaData.allPlatformsDescription, productMediaMetaData.allPlatformsDescriptionPlacement));
        }
    }
    return productMedia;
}
export function mediaPreviewContentFromShelfBasedProductPage(shelfBasedPage) {
    const content = [];
    for (const shelf of Object.values(shelfBasedPage.shelfMapping)) {
        if (productMediaShelf.isProductMediaShelf(shelf)) {
            const productMediaMetaData = shelf.contentsMetadata;
            const productMediaItems = shelf.items;
            const media = new models.ProductMedia(productMediaItems, productMediaMetaData.platform, productMediaMetaData.allPlatforms, productMediaMetaData.platformDescription, productMediaMetaData.allPlatformsDescription, productMediaMetaData.allPlatformsDescriptionPlacement);
            content.push({
                metaData: productMediaMetaData,
                media: media,
            });
        }
    }
    return content;
}
function productReviewsShelfFromShelfBasedProductPage(objectGraph, shelfBasedPage) {
    const productRatingsShelf = shelfBasedPage.shelfMapping["productRatings"];
    const allProductReviewsShelf = shelfBasedPage.shelfMapping["allProductReviews"];
    const editorsChoiceProductReviewsShelf = objectGraph.client.isiOS
        ? shelfBasedPage.shelfMapping["editorsChoice"]
        : shelfBasedPage.shelfMapping["editorsChoiceProductReviews"];
    const tapToRateProductReviewActionShelf = shelfBasedPage.shelfMapping["tapToRateProductReviewAction"];
    const writeAReviewProductReviewActionShelf = shelfBasedPage.shelfMapping["writeAReviewProductReviewAction"];
    if (serverData.isNullOrEmpty(productRatingsShelf) &&
        serverData.isNullOrEmpty(allProductReviewsShelf) &&
        serverData.isNullOrEmpty(editorsChoiceProductReviewsShelf) &&
        serverData.isNullOrEmpty(tapToRateProductReviewActionShelf) &&
        serverData.isNullOrEmpty(writeAReviewProductReviewActionShelf)) {
        return null;
    }
    let editorsChoiceReview = null;
    let userReviews = [];
    let ratings = null;
    let tapToRate = null;
    let writeReviewAction = null;
    let supportAction = null;
    let seeAllAction = null;
    if (serverData.isDefinedNonNullNonEmpty(allProductReviewsShelf === null || allProductReviewsShelf === void 0 ? void 0 : allProductReviewsShelf.items)) {
        for (const item of allProductReviewsShelf.items) {
            const productReview = item;
            switch (productReview.sourceType) {
                case "editorsChoice":
                    editorsChoiceReview = productReview.review;
                    break;
                case "user":
                    userReviews.push(productReview.review);
                    break;
                default:
                    break;
            }
        }
    }
    else {
        userReviews = null;
    }
    if (serverData.isDefinedNonNullNonEmpty(editorsChoiceProductReviewsShelf === null || editorsChoiceProductReviewsShelf === void 0 ? void 0 : editorsChoiceProductReviewsShelf.items)) {
        if (objectGraph.client.isiOS) {
            // On iOS, Editors' Choice is a separate shelf to reviews. On all other platforms, editors choice is embedded in the reviews list.
            editorsChoiceReview = editorsChoiceProductReviewsShelf.items[0];
        }
        else {
            const productReview = editorsChoiceProductReviewsShelf.items[0];
            editorsChoiceReview = productReview.review;
        }
    }
    if (serverData.isDefinedNonNullNonEmpty(tapToRateProductReviewActionShelf === null || tapToRateProductReviewActionShelf === void 0 ? void 0 : tapToRateProductReviewActionShelf.items)) {
        const productReviewAction = tapToRateProductReviewActionShelf.items[0];
        tapToRate = productReviewAction.action;
    }
    if (serverData.isDefinedNonNullNonEmpty(writeAReviewProductReviewActionShelf === null || writeAReviewProductReviewActionShelf === void 0 ? void 0 : writeAReviewProductReviewActionShelf.items)) {
        const productReviewAction = writeAReviewProductReviewActionShelf.items[0];
        const writeAReview = productReviewAction.action;
        writeReviewAction = writeAReview.writeReviewAction;
        supportAction = writeAReview.supportAction;
    }
    if (serverData.isDefinedNonNullNonEmpty(productRatingsShelf === null || productRatingsShelf === void 0 ? void 0 : productRatingsShelf.items)) {
        ratings = productRatingsShelf.items[0];
        seeAllAction = productRatingsShelf === null || productRatingsShelf === void 0 ? void 0 : productRatingsShelf.seeAllAction;
    }
    const reviewsContainer = new models.ReviewsContainer();
    reviewsContainer.adamId = shelfBasedPage.lockup.adamId;
    reviewsContainer.ratings = ratings;
    reviewsContainer.tapToRate = tapToRate;
    reviewsContainer.editorsChoice = editorsChoiceReview;
    reviewsContainer.reviews = userReviews;
    reviewsContainer.supportAction = supportAction;
    reviewsContainer.writeReviewAction = writeReviewAction;
    reviewsContainer.alwaysAllowReviews = shelfBasedPage.alwaysAllowReviews;
    const reviewsShelf = new models.Shelf("reviewsContainer");
    reviewsShelf.title = sectionTitleForPlatform(objectGraph);
    reviewsShelf.items = [reviewsContainer];
    reviewsShelf.seeAllAction = seeAllAction;
    return reviewsShelf;
}
//# sourceMappingURL=product-page.js.map