import { isNothing, isSome } from "@jet/environment";
import { Badge } from "../../../api/models";
import { storefrontContentRatingNameForRank, storefrontContentRatingResourceForRank, } from "../../../common/content/content";
import * as mediaAttributes from "../../../foundation/media/attributes";
import { createArtworkForResource } from "../../content/artwork/artwork";
/**
 * Creates a storefront-specific content badge for storefronts requiring one.
 */
export function createBadge(objectGraph, data, embeddedInRibbon, metricsContext) {
    if (isSome(mediaAttributes.attributeAsDictionary(data, "ageRating"))) {
        // rdar://148625156 (Age Ratings & Assurance: iTMS11 shows double Age Rating badges in ribbon for 10791449607)
        // This is a temporary crutch while iTMS11 is sometimes returning both
        // the old and new age rating models. Sometime after Luck, this whole
        // badge can be removed.
        return null;
    }
    // Australia
    const auContentRank = mediaAttributes.attributeAsNumber(data, "contentRatingsBySystem.appsAustralia.rank");
    const auContentRating = mediaAttributes.attributeAsString(data, "contentRatingsBySystem.appsAustralia.name");
    if (isSome(auContentRank) && isSome(auContentRating)) {
        return createStorefrontContentRatingBadge(objectGraph, embeddedInRibbon, "AU", auContentRank, auContentRating, false);
    }
    // France
    const frContentRank = mediaAttributes.attributeAsNumber(data, "contentRatingsBySystem.appsFrance.rank");
    const frContentRating = mediaAttributes.attributeAsString(data, "contentRatingsBySystem.appsFrance.name");
    if (isSome(frContentRank) && isSome(frContentRating)) {
        return createStorefrontContentRatingBadge(objectGraph, embeddedInRibbon, "FR", frContentRank, frContentRating, false);
    }
    // Korea
    const krContentRank = mediaAttributes.attributeAsNumber(data, "contentRatingsBySystem.appsKorea.rank");
    const krContentRating = mediaAttributes.attributeAsString(data, "contentRatingsBySystem.appsKorea.name");
    if (isSome(krContentRank) && isSome(krContentRating)) {
        return createStorefrontContentRatingBadge(objectGraph, embeddedInRibbon, "KR", krContentRank, krContentRating, true);
    }
    return null;
}
function createStorefrontContentRatingBadge(objectGraph, embeddedInRibbon, storefrontCode, storefrontContentRank, storefrontContentRating, needsFullColor) {
    const pictogramName = storefrontContentRatingNameForRank(objectGraph, storefrontContentRank);
    const pictogramResource = storefrontContentRatingResourceForRank(objectGraph, storefrontContentRank);
    if (isNothing(pictogramResource) || isNothing(pictogramName)) {
        // Return early if there is no artwork to display for the badge. This
        // is the case for the Korea 19+ ranking, or when `rank` is invalid.
        return null;
    }
    const type = "artwork";
    const key = "storefrontContentRating";
    const content = { contentRating: pictogramName };
    const style = objectGraph.client.isTV ? "annotation" : "standard";
    const heading = objectGraph.loc.string(`ProductPage.Badge.AgeRating.${storefrontCode}.Heading`);
    let caption;
    if (embeddedInRibbon && !objectGraph.client.isVision) {
        caption = objectGraph.loc.string("BADGE_AGE_RATING_YEARS");
    }
    else if (objectGraph.client.isVision) {
        caption = objectGraph.loc.string(`ProductPage.Badge.AgeRating.${storefrontCode}.Heading`);
    }
    else {
        caption = objectGraph.loc.string("BADGE_AGE_RATING");
    }
    let axTitle;
    if (objectGraph.client.isTV) {
        // We just want to show the Korea pictogram on TV with nothing after it.
        axTitle = undefined;
    }
    else {
        axTitle = storefrontContentRating;
    }
    const axCaption = objectGraph.loc.string(`ProductPage.Badge.AgeRating.${storefrontCode}.AX.Caption`);
    const badge = new Badge(type, key, content, style, heading, caption, undefined, axTitle, axCaption, "view");
    // Unlike most badges, the Korea pictogram artwork needs to render in full color.
    badge.isMonochrome = !needsFullColor;
    badge.artwork = createArtworkForResource(objectGraph, `resource://${pictogramResource}`, 26, 26);
    return badge;
}
//# sourceMappingURL=storefront-content-rating-badge.js.map