import * as models from "../../../api/models";
import * as artwork from "../../content/artwork/artwork";
import * as productPageUtil from "../product-page-util";
import * as gameController from "./../../content/game-controller";
export function createBadge(objectGraph, data, embeddedInRibbon, metricsContext) {
    var _a;
    if (objectGraph.host.isTV) {
        return createBadgeForTV(objectGraph, data, embeddedInRibbon, metricsContext);
    }
    else {
        return ((_a = createHighPriorityBadge(objectGraph, data, embeddedInRibbon, metricsContext)) !== null && _a !== void 0 ? _a : createLowPriorityBadge(objectGraph, data, embeddedInRibbon, metricsContext));
    }
}
/**
 * Returns a badge that should displayed as higher priority in badge list
 * - In iOS: returns value if badge is `CONTROLLER_RECOMMENDED`
 * - In visionOS: returns value if badge is `CONTROLLER_REQUIRED`
 *
 * This is mutually exclusive with `createLowPriorityBadge`
 */
export function createHighPriorityBadge(objectGraph, data, embeddedInRibbon, metricsContext) {
    if (objectGraph.client.isiOS) {
        return createRecommendedControllerBadgeForIOS(objectGraph, data, embeddedInRibbon, metricsContext);
    }
    else if (objectGraph.client.isVision) {
        return createRequiredControllerBadgeForVisionOS(objectGraph, data, embeddedInRibbon, metricsContext);
    }
    else {
        return null;
    }
}
/**
 * Returns a badge that should displayed as lower priority in badge list (eg: `CONTROLLER_OPTIONAL`)
 *
 * This is mutually exclusive with `createHighPriorityBadge`
 */
export function createLowPriorityBadge(objectGraph, data, embeddedInRibbon, metricsContext) {
    return createSupportedBadge(objectGraph, data, embeddedInRibbon, metricsContext);
}
// MARK: - Private methods
function createBadgeForTV(objectGraph, data, embeddedInRibbon, metricsContext) {
    if (objectGraph.host.isTV) {
        let badge = null;
        const controllerRequirement = gameController.controllerRequirementFromData(objectGraph, data);
        let heading = null;
        let caption = null;
        switch (controllerRequirement) {
            case "SIRI_REMOTE_OR_CONTROLLER_REQUIRED":
                heading = objectGraph.loc.string("BADGE_SIRI_REMOTE_REQUIRED_OR_CONTROLLER_OPTIONAL_HEADING");
                caption = objectGraph.loc.string("BADGE_MFI_SUPPORTED");
                badge = new models.Badge("siriRemoteOrControllerRequired", "siriRemoteOrControllerRequired", {}, "standard", heading, caption, undefined, undefined, caption, "view");
                break;
            case "SIRI_REMOTE_REQUIRED":
                heading = objectGraph.loc.string("BADGE_SIRI_REMOTE_REQUIRED_HEADING");
                caption = objectGraph.loc.string("BADGE_MFI_SUPPORTED");
                badge = new models.Badge("siriRemoteRequired", "siriRemoteRequired", {}, "standard", heading, caption, undefined, undefined, caption, "view");
                break;
            case "CONTROLLER_REQUIRED":
                heading = objectGraph.loc.string("BADGE_MFI_REQUIRED_HEADING");
                caption = objectGraph.loc.string("BADGE_MFI_SUPPORTED");
                badge = new models.Badge("controllerRequired", "controllerRequired", {}, "standard", heading, caption, undefined, undefined, caption, "view");
                break;
            case "CONTROLLER_OPTIONAL":
                heading = objectGraph.loc.string("BADGE_MFI_HEADING");
                caption = objectGraph.loc.string("BADGE_MFI_SUPPORTED");
                badge = new models.Badge("controllerOptional", "controllerOptional", {}, "standard", heading, caption, undefined, undefined, caption, "view");
                break;
            case "NO_BADGE":
                break;
            default:
                break;
        }
        if (badge) {
            badge.clickAction = productPageUtil.isShelfBased(objectGraph)
                ? new models.ShelfBasedPageScrollAction("capabilities")
                : new models.ProductPageScrollAction(new models.ProductPageSection("shelf", "capabilities"));
        }
        return badge;
    }
    else {
        return null;
    }
}
/// This creates a badge only if the controller type is "CONTROLLER_RECOMMENDED", otherwise no badge will be vended.
/// Should use for iOS only as only iOS have `CONTROLLER_RECOMMENDED`
function createRecommendedControllerBadgeForIOS(objectGraph, data, embeddedInRibbon, metricsContext) {
    if (gameController.isGameControllerRecommended(objectGraph, data)) {
        let heading;
        let caption;
        if (!preprocessor.GAMES_TARGET) {
            heading = objectGraph.loc.string("BADGE_MFI_HEADING");
            caption = objectGraph.loc.string("ProductPage.Badge.GameController.Recommended");
        }
        const badge = new models.Badge(createBadgeType(objectGraph), "controller", {}, "standard", heading, caption, undefined, undefined, caption, "view");
        if (preprocessor.GAMES_TARGET) {
            badge.artwork = artwork.createArtworkForResource(objectGraph, "systemimage://gamecontroller.fill");
        }
        else {
            badge.artwork = createAppStoreControllerArtwork(objectGraph);
        }
        badge.clickAction = productPageUtil.isShelfBased(objectGraph)
            ? new models.ShelfBasedPageScrollAction("capabilities")
            : new models.ProductPageScrollAction(new models.ProductPageSection("shelf", "capabilities"));
        return badge;
    }
    else {
        return null;
    }
}
/// This creates a badge only if the controller type is required, otherwise no badge will be vended.
/// Should use for visionOS only.
function createRequiredControllerBadgeForVisionOS(objectGraph, data, embeddedInRibbon, metricsContext) {
    if (gameController.isGameControllerRequired(objectGraph, data)) {
        if (gameController.isSpatialControllerRequired(objectGraph, data)) {
            return null;
        }
        let heading;
        let caption;
        if (!preprocessor.GAMES_TARGET) {
            heading = objectGraph.loc.string("BADGE_MFI_HEADING");
            caption = objectGraph.loc.string("ProductPage.Badge.GameController.Required.Caption.v2");
        }
        const badge = new models.Badge(createBadgeType(objectGraph), "controller", {}, "standard", heading, caption, undefined, undefined, caption, "view");
        badge.artwork = artwork.createArtworkForResource(objectGraph, "systemimage://gamecontroller.fill");
        badge.clickAction = new models.ShelfBasedPageScrollAction("capabilities");
        return badge;
    }
    else {
        return null;
    }
}
/**
 * Creates a badge if a controller is supported (but does not consider the recommended or required case).
 *
 * In iOS, if the feature flag `gameControllerRecommendedRolloutRate` is off, `CONTROLLER_RECOMMENDED` will be consider as supported controller.
 */
function createSupportedBadge(objectGraph, data, embeddedInRibbon, metricsContext) {
    if (gameController.isGameControllerSupported(objectGraph, data) &&
        !gameController.isGameControllerRequired(objectGraph, data) &&
        !gameController.isGameControllerRecommended(objectGraph, data) &&
        !gameController.isSpatialControllerRequired(objectGraph, data)) {
        if (objectGraph.client.isVision && gameController.isSpatialControllerRequired(objectGraph, data)) {
            return null;
        }
        let heading;
        let caption;
        if (!preprocessor.GAMES_TARGET) {
            heading = objectGraph.loc.string("BADGE_MFI_HEADING");
            caption = objectGraph.client.isVision
                ? objectGraph.loc.string("ProductPage.Badge.GameController.Supported.Caption.v2")
                : objectGraph.loc.string("BADGE_MFI_SUPPORTED");
        }
        const badge = new models.Badge(createBadgeType(objectGraph), "controller", {}, "standard", heading, caption, undefined, undefined, caption, "view");
        if (objectGraph.client.isVision || preprocessor.GAMES_TARGET) {
            badge.artwork = artwork.createArtworkForResource(objectGraph, "systemimage://gamecontroller.fill");
        }
        else {
            badge.artwork = createAppStoreControllerArtwork(objectGraph);
        }
        badge.clickAction = productPageUtil.isShelfBased(objectGraph)
            ? new models.ShelfBasedPageScrollAction("capabilities")
            : new models.ProductPageScrollAction(new models.ProductPageSection("shelf", "capabilities"));
        return badge;
    }
    else {
        return null;
    }
}
function createBadgeType(objectGraph) {
    // Be warned, when `artworkControllerBadge` is enabled (as of LuckB), the client no longer supports "controller" as a badge type
    return objectGraph.props.enabled("artworkControllerBadge") ? "artwork" : "controller";
}
function createAppStoreControllerArtwork(objectGraph) {
    const controllerIcon = artwork.createArtworkForResource(objectGraph, "systemimage://gamecontroller.fill");
    controllerIcon.imageScale = "small";
    return controllerIcon;
}
//# sourceMappingURL=controller-badge.js.map