import { isNothing, isSome } from "@jet/environment/types/optional";
import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import * as content from "../../content/content";
import * as metricsHelpersClicks from "../../metrics/helpers/clicks";
import * as productPageUtil from "../product-page-util";
export function badgeArtwork(objectGraph, artworkData, cropCode) {
    // Filter out this badge if no artwork exists
    if (!serverData.isDefinedNonNullNonEmpty(artworkData)) {
        return null;
    }
    let cropCodeToUse = null;
    if (objectGraph.client.isVision) {
        cropCodeToUse = "bb";
    }
    else if (isNothing(cropCode)) {
        cropCodeToUse = "sr";
    }
    else {
        cropCodeToUse = cropCode;
    }
    // Filter out this badge if we can't create an artwork model
    const artwork = content.artworkFromApiArtwork(objectGraph, artworkData, {
        cropCode: cropCodeToUse,
        useCase: 12 /* content.ArtworkUseCase.ProductPageBadge */,
        allowingTransparency: true,
    });
    if (serverData.isNullOrEmpty(artwork)) {
        return null;
    }
    return artwork;
}
/**
 * Contains metadata for an action on a badge. This is useful when the configuration of a badge action is dependent on
 * data in models that might be configured at a later time.
 */
export class BadgeActionMetadata {
    setShelfId(shelfId, index) {
        this.shelfId = shelfId;
        this.index = index;
        if (models.legacyProductPageKnownShelfIds.has(shelfId)) {
            this.section = new models.ProductPageSection("shelf", shelfId);
        }
    }
}
/**
 * Certain parameters on the actions for some badges depend on product page configuration done at a later point
 * in the product page building. This function allows metadata configured at those various parts of the product page
 * build process to be applied.
 *
 * Example: We configure the product page badges in the sidepack, but the action depends on there being an
 * Information section, which is not part of the sidepack. We need the index within the Information section to inform
 * scroll actions exactly which index to scroll to (iOS/macOS), or we need the Information item itself in order to
 * display it (visionOS).
 * @param badges The badges to apply the metadata to.
 * @param badgeActionMetadataByKey The map of badge keys to their respective metadata, if any.
 * @param shelfMetrics The metrics to
 */
export function applyActionMetadataToBadges(objectGraph, badges, badgeActionMetadataByKey, shelfMetrics) {
    var _a, _b, _c, _d, _e;
    let index = 0;
    for (const badge of badges) {
        const metadata = badgeActionMetadataByKey[badge.key];
        if (metadata) {
            let clickAction;
            if (objectGraph.client.isVision) {
                if (isSome(metadata.annotation)) {
                    // We only want to allow presenting an expanded annotation from a ribbon badge if there
                    // is useful extra information to show. This somewhat duplicates logic we have on the
                    // native side, but that logic is specific to annotations in the Information section,
                    // and has some additional checks.
                    const hasSummary = isSome(metadata.annotation.summary) || isSome((_a = metadata.annotation.items[0]) === null || _a === void 0 ? void 0 : _a.text);
                    const hasMultipleItems = metadata.annotation.items.length > 1;
                    const summaryIsDifferentToItem = isSome(metadata.annotation.summary) &&
                        ((_b = metadata.annotation.items[0]) === null || _b === void 0 ? void 0 : _b.text) !== metadata.annotation.summary;
                    const itemHasTextPairs = isSome(((_e = (_d = (_c = metadata.annotation.items[0]) === null || _c === void 0 ? void 0 : _c.textPairs) === null || _d === void 0 ? void 0 : _d.length) !== null && _e !== void 0 ? _e : 0) > 0);
                    const hasLink = isSome(metadata.annotation.linkAction);
                    if (!hasSummary || hasMultipleItems || summaryIsDifferentToItem || itemHasTextPairs || hasLink) {
                        clickAction = metadata.annotation.expandAction;
                    }
                }
            }
            else if (productPageUtil.isShelfBased(objectGraph)) {
                clickAction = new models.ShelfBasedPageScrollAction(metadata.shelfId, null, null, null, metadata.index, true);
            }
            else if (isSome(metadata.section)) {
                const action = new models.ProductPageScrollAction(metadata.section, true);
                action.index = metadata.index;
                clickAction = action;
            }
            badge.clickAction = clickAction;
        }
        let actionType;
        switch (badge.key) {
            case "contentRating":
            case "editorsChoice":
            case "rating":
            case "languages":
            case "size":
            case "controller":
            case "category":
                actionType = "select";
                break;
            default:
                actionType = "navigate";
        }
        if (badge.clickAction) {
            metricsHelpersClicks.addClickEventToAction(objectGraph, badge.clickAction, {
                targetType: "badge",
                id: badge.key,
                idType: "sequential",
                actionType: actionType,
                actionDetails: {
                    position: index,
                },
                pageInformation: shelfMetrics.metricsPageInformation,
                locationTracker: shelfMetrics.locationTracker,
            });
        }
        index += 1;
    }
}
//# sourceMappingURL=badges-common.js.map