import * as models from "../../api/models";
import * as serverData from "../../foundation/json-parsing/server-data";
import * as contentAttributes from "../content/attributes";
import * as metricsHelpersClicks from "../metrics/helpers/clicks";
import { makeRoutableArticlePageCanonicalUrl } from "../today/routable-article-page-url-utils";
import { makeRoutableArticlePageIntent } from "../../api/intents/routable-article-page-intent";
import { getPlatform } from "../preview-platform";
import { getLocale } from "../locale";
/**
 * Determines whether the product has external purchases.
 * @param objectGraph Current object graph
 * @param data The product data
 * @returns True if the product has external purchases
 */
export function hasExternalPurchasesForData(objectGraph, data) {
    const usesExternalPurchase = contentAttributes.contentAttributeAsBooleanOrFalse(objectGraph, data, "usesExternalPurchase");
    const usesExternalLinkPurchase = contentAttributes.contentAttributeAsBooleanOrFalse(objectGraph, data, "usesExternalLinkPurchase");
    return usesExternalPurchase || usesExternalLinkPurchase;
}
/**
 * Determines whether external purchases should be indicated for the given client / bag / placement combination.
 * @param objectGraph Current object graph
 * @param placement Indicates where the external purchases indicator will be placed
 * @returns True if the indicator should be displayed in the given placement.
 */
export function externalPurchasesPlacementIsEnabled(objectGraph, placement) {
    return (objectGraph.bag.enableExternalPurchases &&
        objectGraph.bag.enabledExternalPurchasesPlacements.includes(placement));
}
/**
 * Creates a flow action for the external purchases story.
 * @param objectGraph Current object graph
 * @param title The title to use for the action
 * @param metricsPageInformation Current metrics page information
 * @param metricsLocationTracker Current metrics location tracker
 * @returns A flow action for the story, or null
 */
export function externalPurchasesLearnMoreAction(objectGraph, title, metricsPageInformation, metricsLocationTracker) {
    const editorialItemId = objectGraph.bag.externalPurchasesLearnMoreEditorialItemId;
    if (serverData.isNullOrEmpty(editorialItemId) || !objectGraph.bag.enableExternalPurchases) {
        return null;
    }
    const flowAction = new models.FlowAction("article");
    flowAction.title = title;
    flowAction.pageUrl = `https://apps.apple.com/story/id${editorialItemId}`;
    if (editorialItemId && objectGraph.client.isWeb) {
        const destination = makeRoutableArticlePageIntent({
            ...getLocale(objectGraph),
            ...getPlatform(objectGraph),
            id: editorialItemId,
        });
        const pageUrlString = makeRoutableArticlePageCanonicalUrl(objectGraph, destination);
        flowAction.pageUrl = pageUrlString;
        flowAction.destination = destination;
    }
    metricsHelpersClicks.addClickEventToAction(objectGraph, flowAction, {
        id: "LearnMore",
        targetType: "link",
        actionType: "navigate",
        pageInformation: metricsPageInformation,
        locationTracker: metricsLocationTracker,
    });
    return flowAction;
}
//# sourceMappingURL=external-purchases.js.map