import * as models from "../../../api/models";
import * as mediaRelationship from "../../../foundation/media/relationships";
import { areAppTagsEnabled } from "../../util/app-tags-util";
import { GroupingShelfController } from "./grouping-shelf-controller";
import { attributeAsDictionary } from "@apple-media-services/media-api";
import * as serverData from "../../../foundation/json-parsing/server-data";
import { isSome } from "@jet/environment";
import * as content from "../../content/content";
import { artworkDictionaryFromData } from "../../arcade/breakouts-common";
import * as color from "../../../foundation/util/color-util";
/**
 * This groupings controller adds the logic to display a header shelf on the groupings page. Th primary expereince
 * is artwork and the secondary experience includes fallback icons.
 */
export class GroupingMediaPageHeaderShelfController extends GroupingShelfController {
    // region Constructors
    constructor() {
        super("GroupingMediaPageHeaderShelfController");
        this.supportedFeaturedContentIds = new Set([585 /* groupingTypes.FeaturedContentID.AppStore_MediaPageHeader */]);
    }
    // endregion
    // region Shelf Creation Prerequisites
    /**
     * For a given mediaApiData extract the actual shelfContents array needed to render this shelf
     *
     * @param mediaApiData The outer shelfContents object containing the shelf contents
     */
    initialShelfDataFromGroupingMediaApiData(objectGraph, mediaApiData) {
        return { shelfContents: mediaRelationship.relationshipCollection(mediaApiData, "contents") };
    }
    /**
     * For a given url that this controller handles, we should return a promise that will result in the `ShelfData`
     * needed to render this shelf
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfUrl The url that this controller handled on a secondary fetch
     * @param parameters The extracted parameters from the shelf url
     */
    async secondaryShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters) {
        return await GroupingShelfController.secondaryGroupingShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters);
    }
    /**
     * For a given mediaApiData create an updated shelf token that contains all the additional data for this specific shelf type
     *
     * @param objectGraph The App Store dependency graph
     * @param baseShelfToken The base grouping shelf token created by the grouping-controller
     * @param mediaApiData The outer data object containing the FC properties and data
     * @param groupingParseContext The parse context for the grouping page so far
     */
    shelfTokenFromBaseTokenAndMediaApiData(objectGraph, mediaApiData, baseShelfToken, groupingParseContext) {
        return baseShelfToken;
    }
    // endregion
    // region Shelf Creation
    /**
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfToken The shelf shelfToken for this current shelf creation request
     * @param shelfData The media api shelfContents array for this shelf
     * @param groupingParseContext The parse context used to generate the grouping page on the initial page load,
     * this will be missing when this controller renders a secondary or incomplete shelf fetch.
     */
    _createShelf(objectGraph, shelfToken, shelfData, groupingParseContext) {
        var _a;
        if (!areAppTagsEnabled(objectGraph, "grouping")) {
            return null;
        }
        const headerData = shelfData.shelfContents[0];
        let title;
        if (isSome(headerData)) {
            const editorialNotes = attributeAsDictionary(headerData, "editorialNotes");
            title = (_a = serverData.asString(editorialNotes, "name")) !== null && _a !== void 0 ? _a : null;
            const artworkDictionary = artworkDictionaryFromData(objectGraph, headerData);
            const artworkData = serverData.asDictionary(artworkDictionary, "categoryDetailStatic16x9");
            const artworkFromData = content.artworkFromApiArtwork(objectGraph, artworkData, {
                withJoeColorPlaceholder: true,
                useCase: 0 /* content.ArtworkUseCase.Default */,
                cropCode: "CDS.ApTCHM01",
            });
            if (isSome(title) && isSome(artworkFromData)) {
                const isArtworkDark = color.isDarkColor(artworkFromData.backgroundColor);
                const pageHeader = new models.MediaPageHeader(null, title, null, artworkFromData, null, null, false, null, null, isArtworkDark ? "dark" : "light");
                const shelf = new models.Shelf("mediaPageHeader");
                shelf.items = [pageHeader];
                return shelf;
            }
            return null;
        }
        return null;
    }
}
//# sourceMappingURL=grouping-tags-header-shelf-controller.js.map