import * as serverData from "../../../foundation/json-parsing/server-data";
import * as mediaRelationship from "../../../foundation/media/relationships";
import * as mediaAttributes from "../../../foundation/media/attributes";
import * as groupingTypes from "../grouping-types";
import { createLockupShelf, lockupShelfTokenFromBaseTokenAndMediaApiData, } from "./grouping-lockup-shelf-controller-common";
import { GroupingShelfController } from "./grouping-shelf-controller";
export class GroupingLockupShelfController extends GroupingShelfController {
    // region Constructors
    constructor(builderClass = null) {
        super(builderClass || "GroupingLockupShelfController");
        this.supportedFeaturedContentIds = new Set([
            ...groupingTypes.topChartFeaturedContentIds,
            ...groupingTypes.lockupShelfFeaturedContentIds,
        ]);
    }
    // endregion
    // region Shelf Creation Prerequisites
    /**
     * For a given mediaApiData extract the actual shelfContents array needed to render this shelf
     *
     * @param mediaApiData The outer shelfContents object containing the shelf contents
     */
    initialShelfDataFromGroupingMediaApiData(objectGraph, mediaApiData) {
        const shelfContents = mediaRelationship.relationship(mediaApiData, "contents");
        let shelfData = shelfContents ? shelfContents.data : null;
        if (!shelfData || shelfData.length === 0) {
            shelfData = mediaRelationship.relationshipCollection(mediaApiData, "children");
        }
        return { shelfContents: shelfData };
    }
    /**
     * For a given url that this controller handles, we should return a promise that will result in the `ShelfData`
     * needed to render this shelf
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfUrl The url that this controller handled on a secondary fetch
     * @param parameters The extracted parameters from the shelf url
     */
    async secondaryShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters) {
        return await GroupingShelfController.secondaryGroupingShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters);
    }
    /**
     * For a given mediaApiData create an updated shelf token that contains all the additional data for this specific shelf type
     *
     * @param objectGraph The App Store dependency graph
     * @param baseShelfToken The base grouping shelf token created by the grouping-controller
     * @param mediaApiData The outer data object containing the FC properties and data
     * @param groupingParseContext The parse context for the grouping page so far
     */
    shelfTokenFromBaseTokenAndMediaApiData(objectGraph, mediaApiData, baseShelfToken, groupingParseContext) {
        return lockupShelfTokenFromBaseTokenAndMediaApiData(objectGraph, mediaApiData, baseShelfToken, groupingParseContext);
    }
    // endregion
    // region Metrics
    /**
     * Return the shelf metrics options to use for this specific shelf. Using the base options from the grouping
     * page controller
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfToken The shelf shelfToken for this current shelf creation request
     * @param baseMetricsOptions The minimum set of metrics options for this shelf, created by the
     * grouping page controller
     */
    shelfMetricsOptionsFromBaseMetricsOptions(objectGraph, shelfToken, baseMetricsOptions) {
        const shelfMetricsOptions = { ...baseMetricsOptions };
        shelfMetricsOptions.displayStyle = shelfToken.shelfStyle;
        // Reconfigure Category Breakout metrics options
        if (shelfToken.featuredContentId === 557 /* groupingTypes.FeaturedContentID.AppStore_CategoryBreakoutMarker */) {
            const seeAllContents = mediaRelationship.relationshipData(objectGraph, shelfToken.featuredContentData, "see-all-contents");
            const editorialNotes = mediaAttributes.attributeAsDictionary(seeAllContents, "editorialNotes");
            const title = serverData.asString(editorialNotes, "name");
            shelfMetricsOptions.title = title;
            shelfMetricsOptions.idType = "its_contentId";
            shelfMetricsOptions.badges = { forYou: true };
            shelfMetricsOptions.targetType = "swooshBreakout";
        }
        return shelfMetricsOptions;
    }
    // endregion
    // region Shelf Creation
    /**
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfToken The shelf shelfToken for this current shelf creation request
     * @param shelfData The media api shelfContents array for this shelf
     * @param groupingParseContext The parse context used to generate the grouping page on the initial page load,
     * this will be missing when this controller renders a secondary or incomplete shelf fetch.
     */
    _createShelf(objectGraph, shelfToken, shelfData, groupingParseContext) {
        const isCategoryBreakoutShelf = shelfToken.featuredContentId === 557 /* groupingTypes.FeaturedContentID.AppStore_CategoryBreakoutMarker */;
        if (isCategoryBreakoutShelf) {
            const hasNoContents = !serverData.isDefinedNonNullNonEmpty(shelfData.shelfContents);
            const isClientPad = objectGraph.client.isPad;
            // Drop Category Breakout IF
            // - It has no contents OR
            // - Client is an iPad OR
            if (hasNoContents || isClientPad) {
                return null;
            }
        }
        return createLockupShelf(objectGraph, shelfToken, shelfData, groupingParseContext);
    }
}
//# sourceMappingURL=grouping-lockup-shelf-controller.js.map