import * as models from "../../../api/models";
import * as mediaAttributes from "../../../foundation/media/attributes";
import * as mediaRelationship from "../../../foundation/media/relationships";
import * as todayHorizontalCardUtil from "../../today/today-horizontal-card-util";
import { TodayParseContext } from "../../today/today-types";
import { GroupingShelfController } from "./grouping-shelf-controller";
import * as groupingShelfControllerCommon from "./grouping-shelf-controller-common";
export class GroupingHorizontalCardShelfController extends GroupingShelfController {
    // region Constructors
    constructor() {
        super("GroupingHorizontalCardShelfController");
        this.supportedFeaturedContentIds = new Set([475 /* groupingTypes.FeaturedContentID.AppStore_HorizontalCardSwoosh */]);
    }
    // endregion
    // region GroupingShelfController
    _supports(objectGraph, mediaApiData, featuredContentId, nativeGroupingShelfId) {
        if (!super._supports(objectGraph, mediaApiData, featuredContentId, nativeGroupingShelfId)) {
            return false;
        }
        const displayStyle = mediaAttributes.attributeAsString(mediaApiData, "displayStyle");
        const contentType = groupingShelfControllerCommon.contentTypeForHorizontalCardDisplayStyle(objectGraph, displayStyle);
        return contentType !== "editorialStoryCard";
    }
    // endregion
    // region Shelf Creation Prerequisites
    /**
     * For a given mediaApiData extract the actual shelfContents array needed to render this shelf
     *
     * @param mediaApiData The outer shelfContents object containing the shelf contents
     */
    initialShelfDataFromGroupingMediaApiData(objectGraph, mediaApiData) {
        return { shelfContents: mediaRelationship.relationshipCollection(mediaApiData, "contents") };
    }
    /**
     * For a given url that this controller handles, we should return a promise that will result in the `ShelfData`
     * needed to render this shelf
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfUrl The url that this controller handled on a secondary fetch
     * @param parameters The extracted parameters from the shelf url
     */
    async secondaryShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters) {
        return await GroupingShelfController.secondaryGroupingShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters);
    }
    /**
     * For a given mediaApiData create an updated shelf token that contains all the additional data for this specific shelf type
     *
     * @param objectGraph The App Store dependency graph
     * @param baseShelfToken The base grouping shelf token created by the grouping-controller
     * @param mediaApiData The outer data object containing the FC properties and data
     * @param groupingParseContext The parse context for the grouping page so far
     */
    shelfTokenFromBaseTokenAndMediaApiData(objectGraph, mediaApiData, baseShelfToken, groupingParseContext) {
        const shelfToken = { ...baseShelfToken };
        const displayStyle = mediaAttributes.attributeAsString(mediaApiData, "displayStyle");
        shelfToken.shelfStyle = groupingShelfControllerCommon.contentTypeForHorizontalCardDisplayStyle(objectGraph, displayStyle);
        return shelfToken;
    }
    // endregion
    // region Shelf Creation
    /**
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfToken The shelf shelfToken for this current shelf creation request
     * @param shelfData The media api shelfContents array for this shelf
     * @param groupingParseContext The parse context used to generate the grouping page on the initial page load,
     * this will be missing when this controller renders a secondary or incomplete shelf fetch.
     */
    _createShelf(objectGraph, shelfToken, shelfData, groupingParseContext) {
        var _a;
        const cardUnavailable = function (cardData) {
            shelfToken.remainingItems.push(cardData);
            return false;
        };
        let shelf;
        if (objectGraph.client.isiOS && objectGraph.featureFlags.isEnabled("mini_today_cards_grouping")) {
            const todayParseContext = new TodayParseContext(shelfToken.metricsPageInformation, shelfToken.metricsLocationTracker);
            shelf = todayHorizontalCardUtil.shelfForMiniTodayCards(objectGraph, (_a = shelfData.shelfContents) !== null && _a !== void 0 ? _a : [], shelfToken.title, shelfToken.subtitle, todayParseContext);
        }
        else {
            const cardContext = {
                metricsLocationTracker: shelfToken.metricsLocationTracker,
                metricsPageInformation: shelfToken.metricsPageInformation,
            };
            let resolvedContentType;
            // First check if the client explicitly supports small story cards.
            // If it does, don't change the content type.
            const isSmallStoryCardsSupported = objectGraph.host.isiOS || objectGraph.host.isMac || objectGraph.host.isWeb;
            if (isSmallStoryCardsSupported && shelfToken.shelfStyle === "smallStoryCard") {
                resolvedContentType = shelfToken.shelfStyle;
            }
            else {
                switch (objectGraph.client.deviceType) {
                    case "mac":
                    case "tv":
                    case "web":
                        resolvedContentType = shelfToken.shelfStyle;
                        break;
                    case "watch":
                        resolvedContentType = "todayCard";
                        break;
                    default:
                        resolvedContentType = "todayBrick";
                        break;
                }
            }
            shelf = todayHorizontalCardUtil.shelfForHorizontalCardItems(objectGraph, shelfData.shelfContents, resolvedContentType, shelfToken.title, shelfToken.subtitle, cardContext, cardUnavailable);
            if (shelf.contentType === "smallStoryCard" && Array.isArray(shelf.items)) {
                shelf.items = shelf.items.filter((item) => {
                    if (!(item instanceof models.TodayCard)) {
                        return true;
                    }
                    return todayHorizontalCardUtil.isHorizontalCardSupportedForKind(objectGraph, item.media.kind, "smallStoryCard");
                });
            }
        }
        shelf.url = groupingShelfControllerCommon.createShelfTokenUrlIfNecessaryForShelf(objectGraph, shelf, shelfToken);
        shelf.isHorizontal = true;
        return shelf;
    }
}
//# sourceMappingURL=grouping-horizontal-card-shelf-controller.js.map