import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../foundation/media/attributes";
import * as mediaDataStructure from "../../../foundation/media/data-structure";
import * as mediaRelationship from "../../../foundation/media/relationships";
import * as breakoutsCommon from "../../arcade/breakouts-common";
import * as contentAttributes from "../../content/attributes";
import * as content from "../../content/content";
import * as metricsHelpersClicks from "../../metrics/helpers/clicks";
import * as metricsHelpersImpressions from "../../metrics/helpers/impressions";
import * as metricsHelpersLocation from "../../metrics/helpers/location";
import * as article from "../../today/article";
import * as heroCarouselOverlayCommon from "../hero/hero-carousel-overlay-common";
import * as heroCommon from "../hero/hero-common";
import { GroupingShelfController } from "./grouping-shelf-controller";
import * as groupingShelfControllerCommon from "./grouping-shelf-controller-common";
export class GroupingHeroCarouselShelfController extends GroupingShelfController {
    // region Constructors
    constructor() {
        super("GroupingHeroCarouselShelfController");
        this.supportedFeaturedContentIds = new Set([480 /* groupingTypes.FeaturedContentID.AppStore_Breakout */]);
    }
    // endregion
    // region GroupingShelfController
    _supports(objectGraph, mediaApiData, featuredContentId, nativeGroupingShelfId) {
        if (!super._supports(objectGraph, mediaApiData, featuredContentId, nativeGroupingShelfId)) {
            return false;
        }
        const breakoutStyle = mediaAttributes.attributeAsString(mediaApiData, "displayStyle");
        return breakoutStyle === "hero";
    }
    // endregion
    // region Shelf Creation Prerequisites
    /**
     * For a given mediaApiData extract the actual shelfContents array needed to render this shelf
     *
     * @param mediaApiData The outer shelfContents object containing the shelf contents
     */
    initialShelfDataFromGroupingMediaApiData(objectGraph, mediaApiData) {
        return { shelfContents: mediaRelationship.relationshipCollection(mediaApiData, "contents") };
    }
    /**
     * For a given url that this controller handles, we should return a promise that will result in the `ShelfData`
     * needed to render this shelf
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfUrl The url that this controller handled on a secondary fetch
     * @param parameters The extracted parameters from the shelf url
     */
    async secondaryShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters) {
        return await GroupingShelfController.secondaryGroupingShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters);
    }
    /**
     * For a given mediaApiData create an updated shelf token that contains all the additional data for this specific shelf type
     *
     * @param objectGraph The App Store dependency graph
     * @param baseShelfToken The base grouping shelf token created by the grouping-controller
     * @param mediaApiData The outer data object containing the FC properties and data
     * @param groupingParseContext The parse context for the grouping page so far
     */
    shelfTokenFromBaseTokenAndMediaApiData(objectGraph, mediaApiData, baseShelfToken, groupingParseContext) {
        return baseShelfToken;
    }
    // endregion
    // region Metrics
    shouldImpressShelf() {
        return false;
    }
    // endregion
    // region Shelf Creation
    /**
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfToken The shelf shelfToken for this current shelf creation request
     * @param shelfData The media api shelfContents array for this shelf
     * @param groupingParseContext The parse context used to generate the grouping page on the initial page load,
     * this will be missing when this controller renders a secondary or incomplete shelf fetch.
     */
    _createShelf(objectGraph, shelfToken, shelfData, groupingParseContext) {
        if (groupingParseContext.shelves.length !== 0) {
            return null;
        }
        const shelf = new models.Shelf("heroCarousel");
        const mediaApiData = shelfToken.featuredContentData;
        const heroCarouselMetricsOptions = {
            targetType: "swoosh",
            pageInformation: shelfToken.metricsPageInformation,
            locationTracker: shelfToken.metricsLocationTracker,
            recoMetricsData: mediaDataStructure.metricsFromMediaApiObject(mediaApiData),
        };
        const heroCarousel = new models.HeroCarousel();
        heroCarousel.autoScrollConfiguration = {
            isAutoScrollEnabled: objectGraph.bag.heroCarouselAutoScrollDuration > 0.0,
            autoScrollInterval: objectGraph.bag.heroCarouselAutoScrollDuration,
        };
        const heroCarouselImpressionOptions = metricsHelpersImpressions.impressionOptions(objectGraph, mediaApiData, "heroCarousel", heroCarouselMetricsOptions);
        heroCarouselImpressionOptions.autoAdvanceInterval = heroCarousel.autoScrollConfiguration.autoScrollInterval;
        metricsHelpersImpressions.addImpressionFields(objectGraph, shelf, heroCarouselImpressionOptions);
        metricsHelpersLocation.pushContentLocation(objectGraph, heroCarouselImpressionOptions, "heroCarousel");
        for (const itemData of shelfData.shelfContents) {
            if (serverData.isNull(itemData.attributes) || groupingShelfControllerCommon.shouldDefer(shelfToken)) {
                shelfToken.isDeferring = true;
                shelfToken.remainingItems.push(itemData);
                continue;
            }
            const primaryContent = content.primaryContentForData(objectGraph, itemData);
            if (breakoutsCommon.requiresPrimaryContent(objectGraph, itemData) &&
                !mediaAttributes.hasAttributes(primaryContent)) {
                shelfToken.isDeferring = true;
                shelfToken.remainingItems.push(itemData);
                shelfToken.relationshipToFetch = "primary-content";
                continue;
            }
            const heroCarouselItemMetricsOptions = {
                targetType: "largeBreakout",
                pageInformation: shelfToken.metricsPageInformation,
                locationTracker: shelfToken.metricsLocationTracker,
                recoMetricsData: mediaDataStructure.metricsFromMediaApiObject(itemData),
            };
            const heroVideoData = heroCommon.heroVideoFromData(objectGraph, itemData);
            const heroArtworkData = heroCommon.heroArtworkFromData(objectGraph, itemData);
            if (serverData.isNullOrEmpty(heroVideoData.video) && serverData.isNullOrEmpty(heroArtworkData.artwork)) {
                continue;
            }
            const heroCarouselItem = new models.HeroCarouselItem();
            const productData = article.productDataFromArticle(objectGraph, itemData);
            const metricsTitle = heroCarouselOverlayCommon.heroCarouselItemTitleFromData(objectGraph, itemData);
            const heroCarouselItemImpressionOptions = metricsHelpersImpressions.impressionOptions(objectGraph, itemData, metricsTitle, heroCarouselItemMetricsOptions);
            heroCarouselItemImpressionOptions.isPreorder = contentAttributes.contentAttributeAsBooleanOrFalse(objectGraph, productData, "isPreorder");
            metricsHelpersImpressions.addImpressionFields(objectGraph, heroCarouselItem, heroCarouselItemImpressionOptions);
            // Push the heroCarouselItem here so that the click action has the item in its location
            // but we do not want to add it to the overall location tracker, so pop it right after adding it to the button
            // action
            // <rdar://problem/60883269> Metrics: Arcade: Container values requested in Location field
            metricsHelpersLocation.pushContentLocation(objectGraph, heroCarouselItemImpressionOptions, metricsTitle);
            const titleEffectArtwork = heroVideoData.artworkData || heroArtworkData.artworkData;
            const backgroundColor = heroVideoData.backgroundColor || heroArtworkData.backgroundColor;
            const overlayRequirements = {
                metricsPageInformation: shelfToken.metricsPageInformation,
                metricsLocationTracker: shelfToken.metricsLocationTracker,
                canDisplayArcadeOfferButton: content.shelfContentTypeCanDisplayArcadeOfferButtons(objectGraph, shelfToken.shelfStyle),
                lockupArtworkUseCase: content.artworkUseCaseFromShelfStyle(objectGraph, shelfToken.shelfStyle),
                isContainedInPreorderExclusiveShelf: shelfToken.featuredContentId === 497 /* groupingTypes.FeaturedContentID.AppStore_ComingSoon */,
            };
            heroCarouselItem.overlay = heroCarouselOverlayCommon.overlayFromData(objectGraph, itemData, overlayRequirements);
            heroCarouselItem.backgroundColor = backgroundColor;
            if (!objectGraph.client.isMac || objectGraph.props.isNotEnabled("macOSArcadeHeaderUpdates")) {
                heroCarouselItem.titleEffect = breakoutsCommon.titleEffectFromArtwork(objectGraph, titleEffectArtwork);
            }
            heroCarouselItem.artwork = heroArtworkData.artwork;
            heroCarouselItem.video = heroVideoData.video;
            if (objectGraph.client.isWeb) {
                const portraitVideo = heroCommon.heroVideoFromData(objectGraph, itemData, false, true);
                heroCarouselItem.portraitVideo = portraitVideo === null || portraitVideo === void 0 ? void 0 : portraitVideo.video;
            }
            const heroCarouselItemAction = breakoutsCommon.actionFromData(objectGraph, itemData);
            const heroCarouselItemClickOptions = {
                pageInformation: shelfToken.metricsPageInformation,
                locationTracker: shelfToken.metricsLocationTracker,
                recoMetricsData: mediaDataStructure.metricsFromMediaApiObject(itemData),
                targetType: "largeBreakout",
                id: itemData.id,
            };
            if (heroCarouselItemAction) {
                metricsHelpersClicks.addClickEventToAction(objectGraph, heroCarouselItemAction, heroCarouselItemClickOptions);
                heroCarouselItem.clickAction = heroCarouselItemAction;
            }
            heroCarousel.items.push(heroCarouselItem);
            metricsHelpersLocation.popLocation(heroCarouselItemImpressionOptions.locationTracker);
            metricsHelpersLocation.nextPosition(heroCarouselItemImpressionOptions.locationTracker);
        }
        if (serverData.isDefinedNonNullNonEmpty(heroCarousel.items)) {
            shelf.items = [heroCarousel];
            groupingParseContext.pageTitleEffect = heroCarousel.items[0].titleEffect;
        }
        shelf.url = groupingShelfControllerCommon.createShelfTokenUrlIfNecessaryForShelf(objectGraph, shelf, shelfToken);
        metricsHelpersLocation.popLocation(heroCarouselImpressionOptions.locationTracker);
        metricsHelpersLocation.nextPosition(heroCarouselImpressionOptions.locationTracker);
        return shelf;
    }
}
//# sourceMappingURL=grouping-hero-carousel-shelf-controller.js.map