import * as models from "../../../api/models";
import * as actions from "../../../api/models/actions";
import { Parameters, Path, Protocol } from "../../../foundation/network/url-constants";
import * as urls from "../../../foundation/network/urls";
import { makeGameCenterHeader, openGamesUIAction } from "../../arcade/arcade-common";
import * as metricsHelpersClicks from "../../metrics/helpers/clicks";
import * as metricsHelpersImpressions from "../../metrics/helpers/impressions";
import * as metricsHelpersLocation from "../../metrics/helpers/location";
import { GroupingShelfController } from "./grouping-shelf-controller";
import * as groupingShelfControllerCommon from "./grouping-shelf-controller-common";
export class GroupingGameCenterSuggestedFriendsController extends GroupingShelfController {
    // region Constructors
    constructor() {
        super("GroupingGameCenterSuggestedFriendsController");
        this.batchGroupKey = "gameCenter";
        this.supportedFeaturedContentIds = new Set([496 /* groupingTypes.FeaturedContentID.AppStore_SuggestedFriendsMarker */]);
    }
    // endregion
    // region Shelf Builder
    shelfRoute(objectGraph) {
        return [
            ...super.shelfRoute(objectGraph),
            {
                protocol: Protocol.internal,
                path: `/${Path.grouping}/${Path.shelf}/{token}`,
                query: [Parameters.isGameCenterSuggestedFriendsShelf],
            },
        ];
    }
    // endregion
    // region Shelf Creation Prerequisites
    /**
     * For a given mediaApiData extract the actual shelfContents array needed to render this shelf
     *
     * @param mediaApiData The outer shelfContents object containing the shelf contents
     */
    initialShelfDataFromGroupingMediaApiData(objectGraph, mediaApiData) {
        return {
            $kind: "friendingViaPush",
            shelfContents: [],
            suggestedFriends: [],
        };
    }
    /**
     * For a given url that this controller handles, we should return a promise that will result in the `ShelfData`
     * needed to render this shelf
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfUrl The url that this controller handled on a secondary fetch
     * @param parameters The extracted parameters from the shelf url
     */
    async secondaryShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters) {
        return await objectGraph.gameCenter.fetchSuggestedFriends(10).then((suggestions) => {
            if (objectGraph.props.enabled("gameCenterFriendingViaPush")) {
                return {
                    $kind: "friendingViaPush",
                    shelfContents: [],
                    suggestedFriends: suggestions,
                };
            }
            else {
                return {
                    $kind: "legacy",
                    shelfContents: [],
                    suggestedFriends: suggestions,
                };
            }
        });
    }
    /**
     * For a given mediaApiData create an updated shelf token that contains all the additional data for this specific shelf type
     *
     * @param objectGraph The App Store dependency graph
     * @param baseShelfToken The base grouping shelf token created by the grouping-controller
     * @param mediaApiData The outer data object containing the FC properties and data
     * @param groupingParseContext The parse context for the grouping page so far
     */
    shelfTokenFromBaseTokenAndMediaApiData(objectGraph, mediaApiData, baseShelfToken, groupingParseContext) {
        return baseShelfToken;
    }
    incompleteShelfFetchStrategy(objectGraph) {
        return models.IncompleteShelfFetchStrategy.OnPageLoad;
    }
    // endregion
    // region Shelf Creation
    /**
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfToken The shelf shelfToken for this current shelf creation request
     * @param shelfData The media api shelfContents array for this shelf
     * @param groupingParseContext The parse context used to generate the grouping page on the initial page load,
     * this will be missing when this controller renders a secondary or incomplete shelf fetch.
     */
    _createShelf(objectGraph, shelfToken, shelfData, groupingParseContext) {
        if (shelfToken.isFirstRender) {
            return this.pendingSuggestedFriendsShelfForGrouping(objectGraph, shelfToken, groupingParseContext === null || groupingParseContext === void 0 ? void 0 : groupingParseContext.isArcadePage);
        }
        else {
            return this.suggestedFriendsShelfForGrouping(objectGraph, shelfData, shelfToken, groupingParseContext === null || groupingParseContext === void 0 ? void 0 : groupingParseContext.isArcadePage);
        }
    }
    pendingSuggestedFriendsShelfForGrouping(objectGraph, shelfToken, isArcadePage) {
        if (objectGraph.client.deviceType !== "phone" && objectGraph.client.deviceType !== "pad") {
            return null;
        }
        const shelf = this.suggestedFriendsShelfForGrouping(objectGraph, {
            $kind: objectGraph.props.enabled("gameCenterFriendingViaPush") ? "friendingViaPush" : "legacy",
            shelfContents: [],
            suggestedFriends: [],
        }, shelfToken, isArcadePage);
        const groupingShelfUrl = urls.URL.from(groupingShelfControllerCommon.groupingShelfUrl(shelfToken));
        shelf.url = groupingShelfUrl.param(Parameters.isGameCenterSuggestedFriendsShelf, "true").build();
        shelf.isHidden = shelf.items.length === 0;
        shelf.batchGroup = this.batchGroupKey;
        return shelf;
    }
    suggestedFriendsShelfForGrouping(objectGraph, shelfData, shelfToken, isArcadePage) {
        if (objectGraph.client.deviceType !== "phone" && objectGraph.client.deviceType !== "pad") {
            return null;
        }
        let shelf;
        if (shelfData.$kind === "friendingViaPush") {
            shelf = this.suggestedFriendsShelf(objectGraph, shelfData.suggestedFriends, shelfToken, isArcadePage);
        }
        else {
            shelf = this.legacySuggestedFriendsShelf(objectGraph, shelfData.suggestedFriends, shelfToken, isArcadePage);
        }
        shelf.header = makeGameCenterHeader(objectGraph, shelfToken.title, shelfToken.subtitle);
        shelf.batchGroup = this.batchGroupKey;
        // Hide when empty.
        shelf.isHidden = shelf.items.length === 0;
        shelf.footerTitle = objectGraph.loc.string("Lockup.Footer.GamesApp");
        shelf.footerAction = openGamesUIAction(objectGraph);
        shelf.footerStyle = {
            $kind: "games",
            bundleID: "com.apple.games",
            width: 16,
            height: 16,
        };
        return shelf;
    }
    // region Helpers
    suggestedFriendsShelf(objectGraph, suggestions, token, isArcadePage = false) {
        const suggestionPrefix = "FRIEND_SUGGESTION";
        if (objectGraph.client.deviceType !== "phone" && objectGraph.client.deviceType !== "pad") {
            return null;
        }
        const shelf = new models.Shelf("smallContactCard");
        shelf.isHorizontal = true;
        shelf.mergeWhenFetched = true;
        shelf.batchGroup = "gameCenter";
        const enrichedSuggestions = [];
        for (let index = 0; index < suggestions.length; index++) {
            const suggestionId = `${suggestionPrefix}${index + 1}`;
            const suggestedFriend = suggestions[index];
            const buttonText = objectGraph.loc.string("INVITE");
            const subtitle = objectGraph.loc.string("FROM_CONTACTS");
            const metricsClickOptions = {
                pageInformation: token.metricsPageInformation,
                locationTracker: token.metricsLocationTracker,
                id: suggestionId,
                anonymizationOptions: {
                    anonymizationString: suggestionId,
                },
            };
            let invitationType;
            let shouldShowMessagesBadge;
            if (suggestedFriend.supportsFriendingViaPush && suggestedFriend.contactAssociationID) {
                invitationType = {
                    contact: {
                        contactID: suggestedFriend.contactID,
                        contactAssociationID: suggestedFriend.contactAssociationID,
                    },
                };
                shouldShowMessagesBadge = false;
            }
            else {
                invitationType = {
                    messages: {
                        contactID: suggestedFriend.contactID,
                    },
                };
                shouldShowMessagesBadge = true;
            }
            const buttonAction = new actions.GameCenterInvitePlayerAction(invitationType);
            metricsHelpersClicks.addClickEventToAction(objectGraph, buttonAction, {
                ...metricsClickOptions,
                actionType: "inviteFriend",
            });
            const removeButtonAction = new actions.GameCenterDenylistPlayerAction(suggestedFriend.contactID);
            metricsHelpersClicks.addClickEventToAction(objectGraph, removeButtonAction, {
                ...metricsClickOptions,
                actionType: "removeFriendSuggestion",
            });
            const metricsImpressionOptions = {
                pageInformation: token.metricsPageInformation,
                locationTracker: token.metricsLocationTracker,
                title: suggestionId,
                id: suggestionId,
                kind: "friendSuggestion",
                softwareType: isArcadePage ? "Arcade" : null,
                anonymizationOptions: {
                    anonymizationString: suggestionId,
                },
            };
            const enrichedSuggestion = new models.SmallContactCard(suggestedFriend.contactID, suggestedFriend.fullName, subtitle, buttonText, suggestedFriend.contactID, buttonAction, removeButtonAction, shouldShowMessagesBadge);
            metricsHelpersImpressions.addImpressionFields(objectGraph, enrichedSuggestion, metricsImpressionOptions);
            enrichedSuggestions.push(enrichedSuggestion);
            metricsHelpersLocation.nextPosition(token.metricsLocationTracker);
        }
        shelf.items = enrichedSuggestions;
        shelf.isHidden = shelf.items.length === 0;
        return shelf;
    }
    legacySuggestedFriendsShelf(objectGraph, cards, token, isArcadePage = false) {
        const suggestionPrefix = "FRIEND_SUGGESTION";
        if (objectGraph.client.deviceType !== "phone" && objectGraph.client.deviceType !== "pad") {
            return null;
        }
        const shelf = new models.Shelf("smallContactCard");
        shelf.isHorizontal = true;
        shelf.mergeWhenFetched = true;
        shelf.batchGroup = "gameCenter";
        const enrichedSuggestions = [];
        for (let index = 0; index < cards.length; index++) {
            const suggestionId = `${suggestionPrefix}${index + 1}`;
            const smallContactCard = cards[index];
            smallContactCard.buttonText = objectGraph.loc.string("INVITE");
            smallContactCard.subtitle = objectGraph.loc.string("FROM_CONTACTS");
            const metricsClickOptions = {
                pageInformation: token.metricsPageInformation,
                locationTracker: token.metricsLocationTracker,
                id: suggestionId,
                anonymizationOptions: {
                    anonymizationString: suggestionId,
                },
            };
            smallContactCard.buttonAction = new actions.LegacyGameCenterInvitePlayerAction(smallContactCard.contactId);
            metricsHelpersClicks.addClickEventToAction(objectGraph, smallContactCard.buttonAction, {
                ...metricsClickOptions,
                actionType: "inviteFriend",
            });
            smallContactCard.removeButtonAction = new actions.GameCenterDenylistPlayerAction(smallContactCard.contactId);
            metricsHelpersClicks.addClickEventToAction(objectGraph, smallContactCard.removeButtonAction, {
                ...metricsClickOptions,
                actionType: "removeFriendSuggestion",
            });
            const metricsImpressionOptions = {
                pageInformation: token.metricsPageInformation,
                locationTracker: token.metricsLocationTracker,
                title: suggestionId,
                id: suggestionId,
                kind: "friendSuggestion",
                softwareType: isArcadePage ? "Arcade" : null,
                anonymizationOptions: {
                    anonymizationString: suggestionId,
                },
            };
            metricsHelpersImpressions.addImpressionFields(objectGraph, smallContactCard, metricsImpressionOptions);
            enrichedSuggestions.push(smallContactCard);
            metricsHelpersLocation.nextPosition(token.metricsLocationTracker);
        }
        shelf.items = enrichedSuggestions;
        shelf.isHidden = shelf.items.length === 0;
        return shelf;
    }
    // endregion
    // region Metrics
    shelfMetricsOptionsFromBaseMetricsOptions(objectGraph, shelfToken, baseMetricsOptions) {
        return {
            ...baseMetricsOptions,
            badges: {
                gameCenter: true,
            },
            idType: "its_contentId",
        };
    }
}
//# sourceMappingURL=grouping-game-center-suggested-friends-shelf-controller.js.map