import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../foundation/media/attributes";
import * as mediaRelationship from "../../../foundation/media/relationships";
import { ShelfParameters } from "../../../foundation/network/url-constants";
import * as color from "../../../foundation/util/color-util";
import * as content from "../../content/content";
import * as metricsHelpersLocation from "../../metrics/helpers/location";
import { createTodayBaseCard } from "../../today/cards/today-base-card-builder";
import { TodayParseContext } from "../../today/today-types";
import { GroupingShelfController, routesForFeaturedContentIds } from "./grouping-shelf-controller";
import * as groupingShelfControllerCommon from "./grouping-shelf-controller-common";
export class GroupingEditorialStoryCardShelfController extends GroupingShelfController {
    // region Constructors
    constructor() {
        super("GroupingEditorialStoryCardShelfController");
        this.supportedFeaturedContentIds = new Set([475 /* groupingTypes.FeaturedContentID.AppStore_HorizontalCardSwoosh */]);
    }
    // endregion
    // region Shelf Builder
    shelfRoute(objectGraph) {
        return routesForFeaturedContentIds(this.supportedFeaturedContentIds, [
            `${ShelfParameters.contentType}=editorialStoryCard`,
        ]);
    }
    // endregion
    // region GroupingShelfController
    _supports(objectGraph, mediaApiData, featuredContentId, nativeGroupingShelfId) {
        if (!super._supports(objectGraph, mediaApiData, featuredContentId, nativeGroupingShelfId)) {
            return false;
        }
        const displayStyle = mediaAttributes.attributeAsString(mediaApiData, "displayStyle");
        const contentType = groupingShelfControllerCommon.contentTypeForHorizontalCardDisplayStyle(objectGraph, displayStyle);
        return contentType === "editorialStoryCard";
    }
    // endregion
    // region Shelf Creation Prerequisites
    /**
     * For a given mediaApiData extract the actual shelfContents array needed to render this shelf
     *
     * @param mediaApiData The outer shelfContents object containing the shelf contents
     */
    initialShelfDataFromGroupingMediaApiData(objectGraph, mediaApiData) {
        return { shelfContents: mediaRelationship.relationshipCollection(mediaApiData, "children") };
    }
    /**
     * For a given url that this controller handles, we should return a promise that will result in the `ShelfData`
     * needed to render this shelf
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfUrl The url that this controller handled on a secondary fetch
     * @param parameters The extracted parameters from the shelf url
     */
    async secondaryShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters) {
        return await GroupingShelfController.secondaryGroupingShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters);
    }
    /**
     * For a given mediaApiData create an updated shelf token that contains all the additional data for this specific shelf type
     *
     * @param objectGraph The App Store dependency graph
     * @param baseShelfToken The base grouping shelf token created by the grouping-controller
     * @param mediaApiData The outer data object containing the FC properties and data
     * @param groupingParseContext The parse context for the grouping page so far
     */
    shelfTokenFromBaseTokenAndMediaApiData(objectGraph, mediaApiData, baseShelfToken, groupingParseContext) {
        const shelfToken = { ...baseShelfToken };
        const displayStyle = mediaAttributes.attributeAsString(mediaApiData, "displayStyle");
        shelfToken.shelfStyle = groupingShelfControllerCommon.contentTypeForHorizontalCardDisplayStyle(objectGraph, displayStyle);
        return shelfToken;
    }
    // endregion
    // region Shelf Creation
    /**
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfToken The shelf shelfToken for this current shelf creation request
     * @param shelfData The media api shelfContents array for this shelf
     * @param groupingParseContext The parse context used to generate the grouping page on the initial page load,
     * this will be missing when this controller renders a secondary or incomplete shelf fetch.
     */
    _createShelf(objectGraph, shelfToken, shelfData, groupingParseContext) {
        const items = [];
        for (const card of shelfData.shelfContents) {
            if (!mediaAttributes.hasAttributes(card) || groupingShelfControllerCommon.shouldDefer(shelfToken)) {
                shelfToken.remainingItems.push(card);
                shelfToken.isDeferring = true;
                continue;
            }
            const cardModel = GroupingEditorialStoryCardShelfController.makeStoryCard(objectGraph, card, shelfToken);
            if (serverData.isNullOrEmpty(cardModel)) {
                continue;
            }
            items.push(cardModel);
            metricsHelpersLocation.nextPosition(shelfToken.metricsLocationTracker);
        }
        const shelf = new models.Shelf(shelfToken.shelfStyle);
        shelf.title = shelfToken.title;
        shelf.items = items;
        shelf.isHorizontal = true;
        shelf.background = {
            type: "interactive",
        };
        shelf.url = groupingShelfControllerCommon.createShelfTokenUrlIfNecessaryForShelf(objectGraph, shelf, shelfToken);
        shelf.isHorizontal = true;
        return shelf;
    }
    static makeStoryCard(objectGraph, itemData, shelfToken) {
        // Prefer `subscriptionHero` if it's available for grouping pages, and fallback to `mediaCard` if not.
        let artworkData = mediaAttributes.attributeAsDictionary(itemData, "editorialArtwork.subscriptionHero");
        if (serverData.isNullOrEmpty(artworkData)) {
            artworkData = mediaAttributes.attributeAsDictionary(itemData, "editorialArtwork.mediaCard");
        }
        const artwork = content.artworkFromApiArtwork(objectGraph, artworkData, {
            cropCode: "fn",
            withJoeColorPlaceholder: true,
            useCase: 16 /* content.ArtworkUseCase.TodaySmallStoryCard */,
        });
        if (serverData.isNull(artwork)) {
            return null;
        }
        const title = mediaAttributes.attributeAsString(itemData, "editorialNotes.name");
        const heading = mediaAttributes.attributeAsString(itemData, "label");
        const description = mediaAttributes.attributeAsString(itemData, "editorialNotes.short");
        const cardModel = new models.EditorialStoryCard(title, artwork, null, heading, {
            type: "text",
            title: heading,
        }, description);
        const basicCard = createTodayBaseCard(objectGraph, itemData, null, new TodayParseContext(shelfToken === null || shelfToken === void 0 ? void 0 : shelfToken.metricsPageInformation, shelfToken === null || shelfToken === void 0 ? void 0 : shelfToken.metricsLocationTracker));
        if (serverData.isDefinedNonNull(basicCard)) {
            cardModel.clickAction = basicCard.clickAction;
        }
        const backgroundStyle = color.isDarkColor(artwork.backgroundColor)
            ? "dark"
            : "light";
        cardModel.shelfBackground = {
            type: "artwork",
            artwork: artwork,
            style: backgroundStyle,
        };
        return cardModel;
    }
}
//# sourceMappingURL=grouping-editorial-story-card-shelf-controller.js.map