import { isSome, unwrapOptional as unwrap } from "@jet/environment/types/optional";
import { GenericPage, Shelf, PageHeader } from "../../api/models";
import { makeEditorialShelfCollectionPageIntent, } from "../../api/intents/editorial/editorial-shelf-collection-page-intent";
import { Request, defaultAdditionalPlatformsForClient } from "../../foundation/media/data-fetching";
import { dataFromDataContainer } from "../../foundation/media/data-structure";
import { attributeAsString } from "../../foundation/media/attributes";
import { isDefinedNonNullNonEmpty } from "../../foundation/json-parsing/server-data";
import { metricsPageInformationFromMediaApiResponse, addMetricsEventsToPageWithInformation, } from "../../common/metrics/helpers/page";
import { newLocationTracker } from "../../common/metrics/helpers/location";
import { createBaseShelfToken } from "../../common/editorial-pages/editorial-page-shelf-token";
import { buildEditorialShelf } from "../../common/editorial-pages/editorial-page-shelf-builder";
import { newPageRefreshControllerFromResponse, pageRefreshPolicyForController, } from "../../common/refresh/page-refresh-controller";
import { shouldFetchCustomAttributes } from "../product-page/product-page-variants";
import { prepareMAPIRequest } from "./editorial-page-controller-util";
import { generateRoutes } from "../util/generate-routes";
import { setPreviewPlatform } from "../preview-platform";
import { makeGamesPageHeader } from "../../gameservicesui/src/editorial-page/editorial-component-builder";
import { collectionShelfDisplayStyleFromShelfData } from "./editorial-data-util";
import { collectionDisplayStyleOverride } from "./editorial-page-types";
import { shouldUsePrerenderedIconArtwork } from "../content/content";
// MARK: - `EditorialShelfCollectionPageIntentController` URL Building
const { routes: routesWithoutPlatformSegment, makeCanonicalUrl: makeCanonicalUrlWithoutPlatformSegment } = generateRoutes(makeEditorialShelfCollectionPageIntent, "/collections/{id}");
const { routes: routesWithPlatformSegment, makeCanonicalUrl: makeCanonicalUrlWithPlatformSegment } = generateRoutes(makeEditorialShelfCollectionPageIntent, "/{platform}/collections/{id}");
export function editorialShelfCollectionPageRoutes(objectGraph) {
    return [...routesWithoutPlatformSegment(objectGraph), ...routesWithPlatformSegment(objectGraph)];
}
export function makeEditorialShelfCollectionPageURL(objectGraph, intent) {
    if (intent.platform) {
        return makeCanonicalUrlWithPlatformSegment(objectGraph, intent);
    }
    else {
        return makeCanonicalUrlWithoutPlatformSegment(objectGraph, intent);
    }
}
// MARK: - Media API Request Building
export function defaultRequestAttributes(objectGraph) {
    const attributes = [
        "editorialArtwork",
        "editorialVideo",
        "isAppleWatchSupported",
        "requiredCapabilities",
        "expectedReleaseDateDisplayFormat",
        "showExpectedReleaseDate",
        "badge-content",
        "compatibilityControllerRequirement",
    ];
    if (objectGraph.appleSilicon.isSupportEnabled) {
        attributes.push("macRequiredCapabilities");
    }
    if (objectGraph.client.isMac) {
        attributes.push("hasMacIPAPackage");
    }
    if (objectGraph.bag.enableUpdatedAgeRatings) {
        attributes.push("ageRating");
    }
    if (shouldUsePrerenderedIconArtwork(objectGraph)) {
        attributes.push("iconArtwork");
    }
    return attributes;
}
/**
 * Creates a {@linkcode Request} to fetch a single `editorial-shelves-collection` item by {@linkcode id}
 */
export function makeEditorialShelfCollectionPageRequest(objectGraph, intent) {
    const request = new Request(objectGraph, `/v1/editorial/${intent.storefront}`).withIdOfType(intent.id, "editorial-shelves-collection");
    request
        .includingAgeRestrictions()
        .includingAdditionalPlatforms(defaultAdditionalPlatformsForClient(objectGraph))
        .includingAttributes(defaultRequestAttributes(objectGraph))
        .usingCustomAttributes(shouldFetchCustomAttributes(objectGraph));
    if (objectGraph.client.isWatch) {
        request.addingRelationshipLimit("contents", 3);
    }
    request
        .includingRelationships(["contents"])
        .includingScopedRelationships("editorial-pages", ["primary-contents"])
        .includingAssociateKeys("editorial-shelves-collection:contents", ["editorial-cards"]);
    prepareMAPIRequest(objectGraph, request);
    setPreviewPlatform(objectGraph, request);
    return request;
}
// MARK: - Editorial Shelves Collection Page Rendering
export function renderEditorialShelfCollectionPage(objectGraph, data) {
    var _a;
    const shelfData = dataFromDataContainer(objectGraph, data);
    if (!isDefinedNonNullNonEmpty(shelfData)) {
        return null;
    }
    const metricsPageInformation = metricsPageInformationFromMediaApiResponse(objectGraph, "Room", shelfData.id, data);
    const metricsPageLocationTracker = newLocationTracker();
    const pageRefreshController = newPageRefreshControllerFromResponse(data);
    const collectionDisplayStyle = collectionShelfDisplayStyleFromShelfData(objectGraph, shelfData);
    const shelfToken = createBaseShelfToken(objectGraph, undefined, shelfData, false, 0, metricsPageInformation, metricsPageLocationTracker, preprocessor.GAMES_TARGET, collectionDisplayStyleOverride(collectionDisplayStyle));
    const shelves = [];
    const shelf = unwrap(buildEditorialShelf(objectGraph, undefined, shelfToken));
    shelf.title = null;
    shelf.eyebrow = null;
    shelf.isHorizontal = false;
    const headerTitle = attributeAsString(shelfData, "editorialNotes.name");
    if (isSome(headerTitle)) {
        // There must be a title to create a pageHeader.
        const headerShelf = new Shelf("pageHeader");
        headerShelf.id = "shelf_page_header";
        let pageHeader;
        if (preprocessor.GAMES_TARGET) {
            pageHeader = makeGamesPageHeader(shelfToken, headerTitle, shelf.items, (_a = attributeAsString(shelfData, "editorialNotes.tagline")) !== null && _a !== void 0 ? _a : undefined);
        }
        else {
            pageHeader = new PageHeader(null, headerTitle, attributeAsString(shelfData, "editorialNotes.tagline"));
        }
        headerShelf.items = [pageHeader];
        shelves.push(headerShelf);
    }
    shelves.push(shelf);
    const page = new GenericPage(shelves);
    page.pageRefreshPolicy = pageRefreshPolicyForController(objectGraph, pageRefreshController);
    addMetricsEventsToPageWithInformation(objectGraph, page, metricsPageInformation);
    return page;
}
//# sourceMappingURL=editorial-shelf-collection-page-utils.js.map