import { isNothing, isSome } from "@jet/environment";
import * as models from "../../../api/models";
import { isDefinedNonNullNonEmpty } from "../../../foundation/json-parsing/server-data";
import { attributeAsBooleanOrFalse, attributeAsString } from "../../../foundation/media/attributes";
import { Request } from "../../../foundation/media/data-fetching";
import { relationshipCollection } from "../../../foundation/media/relationships";
import { buildURLFromRequest } from "../../../foundation/media/url-builder";
import { Parameters } from "../../../foundation/network/url-constants";
import { insertPlaceholdersIntoShelfIfRequired } from "../../placeholders/placeholders";
import { CollectionShelfDisplayStyle } from "../editorial-page-types";
import { createShelfSeeAllActionIfNecessary, createShelfTokenUrlIfNecessary, } from "./editorial-page-collection-shelf-builder/editorial-page-collection-shelf-common";
import { buildEditorialLockupHierarchicalPortrait, buildEditorialLockupLargeShelf, buildLockupLargeShelf, buildLockupSmallShelf, } from "./editorial-page-collection-shelf-builder/editorial-page-lockup-collection-shelf-builder";
export function buildRecommendationsShelf(objectGraph, shelfToken) {
    let collectionDisplayStyle = attributeAsString(shelfToken.data, "displayStyle");
    if (preprocessor.GAMES_TARGET) {
        // A way to override display style as design wants to see large cards in "See all" rooms.
        const shelfTokenStyle = shelfToken.collectionDisplayStyle;
        if (isSome(shelfTokenStyle) && shelfTokenStyle !== collectionDisplayStyle) {
            collectionDisplayStyle = shelfToken.collectionDisplayStyle;
        }
    }
    let shelf = null;
    switch (collectionDisplayStyle) {
        case CollectionShelfDisplayStyle.LockupSmall:
            shelf = buildLockupSmallShelf(objectGraph, shelfToken, false);
            break;
        case CollectionShelfDisplayStyle.LockupLarge:
            shelf = buildLockupLargeShelf(objectGraph, shelfToken, false);
            break;
        case CollectionShelfDisplayStyle.EditorialLockupHierarchicalPortrait:
            shelf = buildEditorialLockupHierarchicalPortrait(objectGraph, shelfToken);
            break;
        case CollectionShelfDisplayStyle.EditorialLockupLarge:
            shelf = buildEditorialLockupLargeShelf(objectGraph, shelfToken);
            break;
        default:
            shelf = shelfToken.isFirstRender ? makeHiddenShelf() : null;
            break;
    }
    if (isNothing(shelf)) {
        return null;
    }
    shelf.id = `shelf_${shelfToken.id}`;
    addRecommendationsHrefIfNecessary(objectGraph, shelfToken);
    shelf.url = createShelfTokenUrlIfNecessary(objectGraph, shelf, shelfToken);
    insertPlaceholdersIntoShelfIfRequired(objectGraph, shelf, shelfToken);
    shelf.seeAllAction = createShelfSeeAllActionIfNecessary(objectGraph, shelf, shelfToken);
    return shelf;
}
/**
 * Add the recommendations href for out of band reqeusts if necessary
 * @param shelfToken The shelf token to add the href to
 */
function addRecommendationsHrefIfNecessary(objectGraph, shelfToken) {
    const hasRecommendedContents = isDefinedNonNullNonEmpty(relationshipCollection(shelfToken.data, "contents")) ||
        isDefinedNonNullNonEmpty(shelfToken.remainingItems);
    const isPersonalizationAvailable = attributeAsBooleanOrFalse(shelfToken.data, "isPersonalizationAvailable");
    const baseHref = shelfToken.data.href;
    const shouldAddHref = isPersonalizationAvailable && !hasRecommendedContents && isDefinedNonNullNonEmpty(baseHref);
    if (!shouldAddHref) {
        return;
    }
    const recommendationsRequest = new Request(objectGraph, shelfToken.data.href)
        .addingQuery(Parameters.editorialPageId, shelfToken.pageId)
        .includingRelationships(["contents"]);
    const recommendationsUrl = buildURLFromRequest(objectGraph, recommendationsRequest);
    shelfToken.recommendationsHref = recommendationsUrl.toString();
}
/**
 * Creates a hidden shelf for when we're performing a secondary fetch and need to hide a shelf that returns null
 */
function makeHiddenShelf() {
    const hiddenShelf = new models.Shelf("placeholder");
    hiddenShelf.isHidden = true;
    return hiddenShelf;
}
//# sourceMappingURL=editorial-page-recommendations-shelf-builder.js.map