import * as models from "../../../../api/models";
import * as content from "../../../content/content";
import * as mediaDataStructure from "../../../../foundation/media/data-structure";
import * as metricsHelpersImpressions from "../../../metrics/helpers/impressions";
import * as editorialCopyUtil from "../../editorial-page-editorial-copy-util";
import * as editorialOverlayContentUtil from "../../editorial-page-overlay-content-util";
import { asBooleanOrFalse, isDefinedNonNull, isDefinedNonNullNonEmpty, } from "../../../../foundation/json-parsing/server-data";
import * as editorialDataUtil from "../../editorial-data-util";
import * as editorialActionUtil from "../../editorial-action-util";
import * as editorialMediaUtil from "../../editorial-media-util";
import * as mediaAttributes from "../../../../foundation/media/attributes";
import { deepLinkUrlFromData } from "../../../linking/external-deep-link";
import { shouldFilter } from "../../../filtering";
import { mixedMediaItemFromEditorialData } from "../../../../gameservicesui/src/common/mixed-media-item-editorial-artwork";
/**
 * Build an `EditorialStoryCard` from the provided data.
 * @param objectGraph The object graph.
 * @param itemData Data for the card item being created.
 * @param mediaPlacement The placement within which the card is being created for media.
 * @param metricsOptions A set of metrics options.
 * @param collectionDisplayStyle The collection display style for the shelf this is in.
 * @param allowSuppressLockup Whether any lockup can be suppressed
 * @param shelfToken A shelf token, if being presented within a shelf. If not, it's being used in another context.
 * @returns The built editorial story card.
 */
export function buildStoryCard(objectGraph, itemData, mediaPlacement, metricsOptions, collectionDisplayStyle, allowSuppressLockup, shelfToken) {
    var _a, _b;
    const editorialClientParams = editorialDataUtil.extractEditorialClientParams(objectGraph, itemData);
    const storyCard = new models.EditorialStoryCard();
    storyCard.title = editorialCopyUtil.editorialTitleFromData(objectGraph, itemData, collectionDisplayStyle);
    if (!editorialClientParams.suppressShort) {
        storyCard.description = editorialCopyUtil.editorialDescriptionFromData(objectGraph, itemData);
    }
    const hideEditorialMedia = asBooleanOrFalse(editorialClientParams.hideEditorialMedia);
    if (!hideEditorialMedia) {
        const mediaData = editorialMediaUtil.editorialMediaDataForPlacement(objectGraph, itemData, mediaPlacement);
        storyCard.artwork = mediaData === null || mediaData === void 0 ? void 0 : mediaData.artwork;
        storyCard.video = mediaData === null || mediaData === void 0 ? void 0 : mediaData.video;
        storyCard.isMediaDark = (_a = editorialMediaUtil.isMediaDark(objectGraph, mediaData)) !== null && _a !== void 0 ? _a : undefined;
        if (mediaPlacement === editorialMediaUtil.EditorialMediaPlacement.StoryDetail) {
            if (preprocessor.GAMES_TARGET) {
                storyCard.mixedMediaItem = mixedMediaItemFromEditorialData(objectGraph, itemData, "DetailStory");
            }
            const landscapeMediaData = editorialMediaUtil.editorialMediaDataForPlacement(objectGraph, itemData, editorialMediaUtil.EditorialMediaPlacement.StoryDetailLandscape);
            storyCard.landscapeArtwork = landscapeMediaData === null || landscapeMediaData === void 0 ? void 0 : landscapeMediaData.artwork;
            storyCard.landscapeVideo = landscapeMediaData === null || landscapeMediaData === void 0 ? void 0 : landscapeMediaData.video;
            storyCard.isLandscapeMediaDark =
                (_b = editorialMediaUtil.isMediaDark(objectGraph, landscapeMediaData)) !== null && _b !== void 0 ? _b : undefined;
        }
    }
    const lockupOptions = {
        metricsOptions: {
            pageInformation: metricsOptions.pageInformation,
            locationTracker: metricsOptions.locationTracker,
            recoMetricsData: mediaDataStructure.metricsFromMediaApiObject(itemData),
        },
        artworkUseCase: 1 /* content.ArtworkUseCase.LockupIconSmall */,
        offerEnvironment: editorialClientParams.useMaterialBlur ? "light" : "lightOverArtwork",
        canDisplayArcadeOfferButton: content.shelfDisplayStyleCanDisplayArcadeOfferButtons(objectGraph, collectionDisplayStyle),
        isContainedInPreorderExclusiveShelf: false,
        externalDeepLinkUrl: deepLinkUrlFromData(objectGraph, itemData),
        useJoeColorIconPlaceholder: preprocessor.GAMES_TARGET,
    };
    const overlayContent = editorialOverlayContentUtil.extractOverlayContent(objectGraph, itemData, lockupOptions);
    const suppressLockup = allowSuppressLockup && asBooleanOrFalse(editorialClientParams.suppressLockup);
    if (!suppressLockup) {
        storyCard.lockup = overlayContent.lockup;
    }
    // For Arcade app that is preorder and bincompat, we should override the badge to always display "COMING SOON" in upper cased.
    const lockupData = editorialDataUtil.extractProductData(objectGraph, itemData);
    const isPreorder = mediaAttributes.attributeAsBooleanOrFalse(lockupData, "isPreorder");
    const isArcadeLockup = content.isArcadeSupported(objectGraph, lockupData);
    const supportsVisionOSCompatibleIOSBinary = content.supportsVisionOSCompatibleIOSBinaryFromData(objectGraph, lockupData);
    // If there is lockup data, and we're not suppressing the lockup, check the data is valid for this platform.
    // If this is for a story detail page, ignore these checks so that the artwork card is still presented as expected.
    if (isDefinedNonNullNonEmpty(lockupData) &&
        !suppressLockup &&
        shouldFilter(objectGraph, lockupData, 80894 /* Filter.All */) &&
        mediaPlacement !== editorialMediaUtil.EditorialMediaPlacement.StoryDetail) {
        return null;
    }
    let badgeText;
    if (isPreorder && isArcadeLockup && supportsVisionOSCompatibleIOSBinary) {
        badgeText = objectGraph.loc.string("ARCADE_PREORDER_COMING_SOON");
    }
    else {
        badgeText = editorialCopyUtil.editorialBadgeFromData(objectGraph, itemData, collectionDisplayStyle);
    }
    if (objectGraph.client.isVision && badgeText === objectGraph.loc.string("APPLE_ARCADE")) {
        storyCard.badge = {
            type: "arcadeWordmark",
        };
    }
    else {
        storyCard.badge = {
            type: "text",
            title: badgeText,
        };
    }
    storyCard.collectionIcons = overlayContent.collectionIcons;
    if (isDefinedNonNull(shelfToken)) {
        const baseClickOptions = {
            id: itemData.id,
            pageInformation: shelfToken.metricsPageInformation,
            locationTracker: shelfToken.metricsLocationTracker,
            actionDetails: {
                franchise: storyCard.badge.title,
            },
            idType: "editorial_id",
        };
        storyCard.clickAction = editorialActionUtil.createPrimaryActionForComponentFromData(objectGraph, itemData, shelfToken, baseClickOptions);
    }
    storyCard.editorialDisplayOptions =
        editorialDataUtil.editorialDisplayOptionsFromClientParams(editorialClientParams);
    // Configure impressions
    const contentMetricsOptions = {
        ...metricsOptions,
        id: itemData.id,
        idType: "editorial_id",
    };
    const impressionOptions = metricsHelpersImpressions.impressionOptions(objectGraph, itemData, storyCard.title, contentMetricsOptions);
    metricsHelpersImpressions.addImpressionFields(objectGraph, storyCard, impressionOptions);
    if (!storyCard.isValid()) {
        return null;
    }
    return storyCard;
}
//# sourceMappingURL=editorial-page-story-card-utils.js.map