import { isSome } from "@jet/environment";
import * as models from "../../../../api/models";
import * as serverData from "../../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../../foundation/media/attributes";
import * as filtering from "../../../filtering";
import * as metricsHelpersLocation from "../../../metrics/helpers/location";
import { insertPlaceholdersIntoShelfIfRequired } from "../../../placeholders/placeholders";
import * as editorialDataUtil from "../../editorial-data-util";
import { lockupFromData } from "../../editorial-page-lockup-utils";
import { CollectionShelfDisplayStyle, CollectionShelfFilterOverride, collectionShelfLayoutDirection, CollectionShelfLayoutDirection, displayLimitForCollectionShelf, } from "../../editorial-page-types";
import { createShelfTokenUrlIfNecessary, shouldDefer } from "./editorial-page-collection-shelf-common";
import * as gamesComponentBuilder from "../../../../gameservicesui/src/editorial-page/editorial-component-builder";
import * as charts from "../../../../gameservicesui/src/utility/charts";
import { arcadeColor } from "../../../arcade/arcade-common";
import { artworkTemplateForBundleImage, createArtworkForResource } from "../../../content/artwork/artwork";
export function buildEditorialLockupHierarchicalPortrait(objectGraph, shelfToken) {
    return buildLockupShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.EditorialLockupHierarchicalPortrait, "mediumImageLockup", true, CollectionShelfLayoutDirection.Vertical);
}
export function buildEditorialLockupHierarchicalRows(objectGraph, shelfToken) {
    return buildLockupShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.EditorialLockupHierarchicalRows, "mediumImageLockup", true, CollectionShelfLayoutDirection.Vertical);
}
export function buildEditorialLockupMediumShelf(objectGraph, shelfToken) {
    return buildLockupShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.EditorialLockupMedium, "mediumImageLockup");
}
export function buildEditorialLockupMediumVariantShelf(objectGraph, shelfToken) {
    return buildLockupShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.EditorialLockupMediumVariant, "mediumImageLockup");
}
export function buildEditorialLockupLargeShelf(objectGraph, shelfToken) {
    return buildLockupShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.EditorialLockupLarge, "largeImageLockup");
}
export function buildEditorialLockupLargeVariantShelf(objectGraph, shelfToken) {
    return buildLockupShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.EditorialLockupLargeVariant, "largeImageLockup");
}
export function buildLockupChartShelf(objectGraph, shelfToken) {
    return buildLockupShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.Charts, shelfToken.isSeeAll ? "smallLockup" : "mediumImageLockup", true, CollectionShelfLayoutDirection.Vertical);
}
export function buildLockupMedium(objectGraph, shelfToken) {
    shelfToken.metricsImpressionOptions.shouldOmitImpressionIndex = true;
    return buildLockupShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.LockupMedium, "mediumLockup");
}
export function buildLockupSmallShelf(objectGraph, shelfToken, insertPlaceholdersIfRequired = true, overrideShelfLayoutDirection) {
    if (objectGraph.featureFlags.isEnabled("force_display_lockup_ordinals")) {
        // For 2023F Charts testing before server is ready. Can be removed in the 2024A timeline.
        shelfToken.showOrdinals = true;
    }
    return buildLockupShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.LockupSmall, "smallLockup", insertPlaceholdersIfRequired, overrideShelfLayoutDirection);
}
export function buildLockupLargeShelf(objectGraph, shelfToken, insertPlaceholdersIfRequired = true) {
    return buildLockupShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.LockupLarge, "largeLockup", insertPlaceholdersIfRequired);
}
export function buildEditorialLockupPosterShelf(objectGraph, shelfToken) {
    const shelf = buildLockupShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.Poster, "posterLockup");
    if (isSome(shelf)) {
        shelf.horizontalScrollTargetBehavior = models.ShelfHorizontalScrollTargetBehavior.CenterAligned;
    }
    return shelf;
}
// region Lockup Creation
function buildLockupShelf(objectGraph, shelfToken, displayStyle, lockupComponentType, insertPlaceholdersIfRequired = true, overrideShelfLayoutDirection) {
    const items = [];
    const shelf = new models.Shelf(lockupComponentType);
    const layoutDirection = overrideShelfLayoutDirection !== null && overrideShelfLayoutDirection !== void 0 ? overrideShelfLayoutDirection : collectionShelfLayoutDirection(shelfToken);
    shelf.isHorizontal = layoutDirection === CollectionShelfLayoutDirection.Horizontal;
    const shelfContents = editorialDataUtil.extractRemainingItemsOrInitialShelfContents(objectGraph, shelfToken);
    const displayLimit = displayLimitForCollectionShelf(displayStyle, layoutDirection, shelfToken);
    for (const [index, itemData] of shelfContents.entries()) {
        // If we encounter a type of app-events, this means they have been incorrectly programmed,
        // and we should throw the shelf away.
        if (itemData.type === "app-events") {
            return null;
        }
        if (serverData.isNull(itemData.attributes) || shouldDefer(shelfToken)) {
            shelfToken.isDeferring = true;
            shelfToken.remainingItems.push(itemData);
            continue;
        }
        // Filter out unwanted content
        if (filtering.shouldFilter(objectGraph, itemData, 80894 /* filtering.Filter.All */)) {
            continue;
        }
        let lockup = null;
        if (preprocessor.GAMES_TARGET && gamesComponentBuilder.shouldUseChartCard(displayStyle, shelfToken)) {
            lockup = gamesComponentBuilder.makeChartCard(objectGraph, itemData, shelfToken, index, "Editorial");
        }
        else if (preprocessor.GAMES_TARGET && gamesComponentBuilder.shouldUseMediaCard(displayStyle)) {
            lockup = gamesComponentBuilder.makeEditorialMediaCard(objectGraph, itemData, "Card", shelfToken, "Editorial");
        }
        else if (preprocessor.GAMES_TARGET && gamesComponentBuilder.shouldUseGameLockup(displayStyle)) {
            lockup = gamesComponentBuilder.makeGameLockupShelfModel(objectGraph, itemData, shelfToken);
        }
        else {
            lockup = lockupFromData(objectGraph, itemData, shelfToken, items.length, displayStyle);
        }
        if (lockup) {
            items.push(lockup);
            metricsHelpersLocation.nextPosition(shelfToken.metricsImpressionOptions.locationTracker);
            shelfToken.ordinalIndex++;
            if (items.length === displayLimit) {
                break;
            }
        }
    }
    if (displayStyle === CollectionShelfDisplayStyle.EditorialLockupMediumVariant ||
        displayStyle === CollectionShelfDisplayStyle.EditorialLockupLargeVariant) {
        if (serverData.isNull(shelf.presentationHints)) {
            shelf.presentationHints = { showSupplementaryText: true };
        }
        else {
            shelf.presentationHints = {
                ...shelf.presentationHints,
                showSupplementaryText: true,
            };
        }
    }
    // We don't need this in our incomplete shelf URL, so we'll preemptively remove it.
    delete shelfToken.maxItemCount;
    shelf.eyebrow = shelfToken.eyebrow;
    shelf.eyebrowArtwork = shelfToken.eyebrowArtwork;
    shelf.title = shelfToken.title;
    shelf.subtitle = shelfToken.subtitle;
    // Hydrate Games data if the eyebrow has not already been populated from the shelf token.
    if (preprocessor.GAMES_TARGET && shelf.eyebrow == null && shelf.eyebrowArtwork == null) {
        populateGamesHeaderContent(shelf, shelfToken, objectGraph);
    }
    if (preprocessor.GAMES_TARGET &&
        gamesComponentBuilder.shouldUseComponentGrid(displayStyle, layoutDirection, shelfToken)) {
        const item = gamesComponentBuilder.makeComponentGrid(objectGraph, items, shelfToken, displayStyle);
        shelf.items = isSome(item) ? [item] : [];
    }
    else {
        shelf.items = items;
    }
    shelf.rowsPerColumn = mediaAttributes.attributeAsNumber(shelfToken.data, "rowCount");
    shelf.shouldFilterApps = !shelfToken.filterOverrides.includes(CollectionShelfFilterOverride.ShowInstalled);
    shelf.url = createShelfTokenUrlIfNecessary(objectGraph, shelf, shelfToken);
    if (insertPlaceholdersIfRequired) {
        let placeholderDisplayLimit = displayLimit;
        if (isSome(displayLimit) &&
            displayLimit > 4 &&
            (displayStyle === CollectionShelfDisplayStyle.Charts ||
                displayStyle === CollectionShelfDisplayStyle.EditorialLockupMedium)) {
            // TODO: Remove this override when this radar is complete: rdar://148395979 ([EditorialPage] Placeholders for Charts and 4UP are not rendering as a ComponentGrid)
            placeholderDisplayLimit = 4;
        }
        insertPlaceholdersIntoShelfIfRequired(objectGraph, shelf, shelfToken, placeholderDisplayLimit);
    }
    return shelf;
}
/**
 * Hydrates game related header data
 *
 * Note that both the header and non header fields should be hydrated, since header
 * preference can be switched off. In this case we will not have the capacity to color the Arcade eyebrow
 */
function populateGamesHeaderContent(shelf, shelfToken, objectGraph) {
    switch (serverData.asString(shelfToken.data.attributes, "chart")) {
        case "combined-game-center" /* TopChartType.CombinedGameCenter */:
            const gameCenterEyebrow = charts.makeChartsTopSubtitle("game-center", objectGraph);
            const gameCenterEyebrowArtwork = createArtworkForResource(objectGraph, artworkTemplateForBundleImage("ProductCapabilityGameCenter"));
            shelf.eyebrow = gameCenterEyebrow;
            shelf.eyebrowArtwork = gameCenterEyebrowArtwork;
            shelf.header = {
                eyebrow: gameCenterEyebrow,
                eyebrowArtwork: gameCenterEyebrowArtwork,
                eyebrowArtworkSize: 14,
                title: shelfToken.title,
                subtitle: shelfToken.subtitle,
            };
            break;
        case "top-arcade" /* TopChartType.TopArcade */:
            const color = arcadeColor;
            const arcadeEyebrow = charts.makeChartsTopSubtitle("arcade", objectGraph);
            shelf.eyebrow = arcadeEyebrow;
            shelf.header = {
                eyebrow: arcadeEyebrow,
                title: shelfToken.title,
                subtitle: shelfToken.subtitle,
                configuration: {
                    eyebrowColor: {
                        type: "rgb",
                        red: color.red,
                        green: color.green,
                        blue: color.blue,
                    },
                },
            };
            break;
        default:
            break;
    }
}
// endregion
//# sourceMappingURL=editorial-page-lockup-collection-shelf-builder.js.map