import * as models from "../../../../api/models";
import * as serverData from "../../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../../foundation/media/attributes";
import * as mediaDataStructure from "../../../../foundation/media/data-structure";
import { shallowCopyOf } from "../../../../foundation/util/objects";
import * as breakoutsCommon from "../../../arcade/breakouts-common";
import * as contentAttributes from "../../../content/attributes";
import * as metricsHelpersImpressions from "../../../metrics/helpers/impressions";
import * as metricsHelpersLocation from "../../../metrics/helpers/location";
import * as article from "../../../today/article";
import * as editorialDataUtil from "../../editorial-data-util";
import * as editorialMediaUtil from "../../editorial-media-util";
import { editorialTitleFromData } from "../../editorial-page-editorial-copy-util";
import { heroOverlayFromData } from "../../editorial-page-hero-util";
import * as editorialOverlayContentUtil from "../../editorial-page-overlay-content-util";
import { CollectionShelfDisplayStyle } from "../../editorial-page-types";
import { createShelfTokenUrlIfNecessary, shouldDefer } from "./editorial-page-collection-shelf-common";
export function buildHeroShelf(objectGraph, shelfToken) {
    if (shelfToken.shelfIndex !== 0) {
        return null;
    }
    const shelf = new models.Shelf("heroCarousel");
    const heroCarousel = new models.HeroCarousel();
    heroCarousel.autoScrollConfiguration = heroAutoscrollConfiguration(objectGraph);
    const shelfContents = editorialDataUtil.extractRemainingItemsOrInitialShelfContents(objectGraph, shelfToken);
    for (const itemData of shelfContents) {
        const primaryItemData = editorialDataUtil.extractHydratedPrimaryContentData(objectGraph, itemData);
        if ((editorialDataUtil.requiresPrimaryContent(objectGraph, itemData) &&
            !mediaAttributes.hasAttributes(primaryItemData)) ||
            shouldDefer(shelfToken)) {
            shelfToken.isDeferring = true;
            shelfToken.remainingItems.push(itemData);
            continue;
        }
        const heroCarouselItemMetricsOptions = {
            ...shelfToken.metricsImpressionOptions,
            idType: "its_contentId",
            targetType: "hero",
            pageInformation: shelfToken.metricsPageInformation,
            locationTracker: shelfToken.metricsLocationTracker,
            recoMetricsData: mediaDataStructure.metricsFromMediaApiObject(itemData),
        };
        const heroMediaData = editorialMediaUtil.editorialMediaDataForPlacement(objectGraph, itemData, editorialMediaUtil.EditorialMediaPlacement.Hero);
        const lockupOptions = {
            metricsOptions: heroCarouselItemMetricsOptions,
            clientIdentifierOverride: shelfToken.clientIdentifierOverride,
            artworkUseCase: 0 /* content.ArtworkUseCase.Default */,
            offerEnvironment: "lightOverArtwork",
            canDisplayArcadeOfferButton: true,
        };
        const overlayContent = editorialOverlayContentUtil.extractOverlayContent(objectGraph, itemData, lockupOptions);
        const heroCarouselItem = new models.HeroCarouselItem();
        const productData = article.productDataFromArticle(objectGraph, itemData);
        const metricsTitle = editorialTitleFromData(objectGraph, itemData, CollectionShelfDisplayStyle.Hero);
        const heroCarouselItemImpressionOptions = metricsHelpersImpressions.impressionOptions(objectGraph, itemData, metricsTitle, heroCarouselItemMetricsOptions);
        heroCarouselItemImpressionOptions.isPreorder = contentAttributes.contentAttributeAsBooleanOrFalse(objectGraph, productData, "isPreorder");
        metricsHelpersImpressions.addImpressionFields(objectGraph, heroCarouselItem, heroCarouselItemImpressionOptions);
        // Push the heroCarouselItem here so that the click action has the item in its location
        // but we do not want to add it to the overall location tracker, so pop it right after adding it to the button
        // actiono
        // <rdar://problem/60883269> Metrics: Arcade: Container values requested in Location field
        metricsHelpersLocation.pushContentLocation(objectGraph, heroCarouselItemImpressionOptions, metricsTitle);
        const editorialClientParams = editorialDataUtil.extractEditorialClientParams(objectGraph, itemData);
        const hideEditorialMedia = serverData.asBooleanOrFalse(editorialClientParams.hideEditorialMedia);
        heroCarouselItem.overlay = heroOverlayFromData(objectGraph, itemData, shelfToken, editorialClientParams);
        heroCarouselItem.collectionIcons = overlayContent.collectionIcons;
        heroCarouselItem.editorialDisplayOptions =
            editorialDataUtil.editorialDisplayOptionsFromClientParams(editorialClientParams);
        populateHeroCarouselItemMedia(objectGraph, heroCarouselItem, hideEditorialMedia, heroMediaData, false);
        if (heroCarouselItem.isValid()) {
            heroCarousel.items.push(heroCarouselItem);
        }
        const rtlHeroCarouselItem = shallowCopyOf(heroCarouselItem);
        populateHeroCarouselItemMedia(objectGraph, rtlHeroCarouselItem, hideEditorialMedia, heroMediaData, true);
        if (rtlHeroCarouselItem.isValid()) {
            heroCarousel.rtlItems.push(rtlHeroCarouselItem);
        }
        metricsHelpersLocation.popLocation(heroCarouselItemImpressionOptions.locationTracker);
        metricsHelpersLocation.nextPosition(heroCarouselItemImpressionOptions.locationTracker);
    }
    // We don't need this in our incomplete shelf URL, so we'll preemptively remove it.
    delete shelfToken.maxItemCount;
    shelf.isHorizontal = false;
    shelf.eyebrow = shelfToken.eyebrow;
    shelf.eyebrowArtwork = shelfToken.eyebrowArtwork;
    shelf.title = shelfToken.title;
    shelf.subtitle = shelfToken.subtitle;
    shelf.items = [heroCarousel];
    shelf.url = createShelfTokenUrlIfNecessary(objectGraph, shelf, shelfToken);
    return shelf;
}
/**
 * Populates the media properties of a `HeroCarouselItem`.
 * @param objectGraph Current object graph
 * @param heroCarouselItem The item to populate
 * @param hideEditorialMedia Whether to hide editorial media
 * @param heroMediaData The `EditorialMediaData` to derive the media properties from
 * @param isRTL Whether this item will be used in an RTL layout
 */
function populateHeroCarouselItemMedia(objectGraph, heroCarouselItem, hideEditorialMedia, heroMediaData, isRTL) {
    if (!hideEditorialMedia) {
        const backgroundColor = isRTL ? heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.rtlBackgroundColor : heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.backgroundColor;
        heroCarouselItem.backgroundColor = backgroundColor;
        const artworkData = isRTL ? heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.rtlArtworkData : heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.artworkData;
        if (!objectGraph.client.isMac || objectGraph.props.isNotEnabled("macOSArcadeHeaderUpdates")) {
            heroCarouselItem.titleEffect = breakoutsCommon.titleEffectFromArtwork(objectGraph, artworkData);
        }
        const artwork = isRTL ? heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.rtlArtwork : heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.artwork;
        heroCarouselItem.artwork = artwork;
        const video = isRTL ? heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.rtlVideo : heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.video;
        heroCarouselItem.video = video;
    }
    heroCarouselItem.isMediaDark = editorialMediaUtil.isMediaDark(objectGraph, heroMediaData, isRTL);
}
/**
 * Generates the auto scroll configuration for the hero carousel
 * @param objectGraph Current object graph
 * @returns Built auto scroll configuration
 */
export function heroAutoscrollConfiguration(objectGraph) {
    return {
        isAutoScrollEnabled: objectGraph.bag.heroCarouselAutoScrollDuration > 0.0,
        autoScrollInterval: objectGraph.bag.heroCarouselAutoScrollDuration,
    };
}
/**
 * Decorates the input shelf metrics options with shelf specific fields.
 * @param objectGraph Current object graph
 * @param shelfMetricsOptions Base shelf metrics options
 * @returns Decorated shelf metrics options
 */
export function decorateHeroShelfMetricsOptions(objectGraph, shelfMetricsOptions) {
    shelfMetricsOptions.title = "heroCarousel";
    const autoScrollConfiguration = heroAutoscrollConfiguration(objectGraph);
    shelfMetricsOptions.autoAdvanceInterval = autoScrollConfiguration.autoScrollInterval;
}
//# sourceMappingURL=editorial-page-hero-collection-shelf-builder.js.map