import { isNothing, isSome } from "@jet/environment/types/optional";
import * as models from "../../../../api/models";
import * as serverData from "../../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../../foundation/media/attributes";
import * as mediaDataStructure from "../../../../foundation/media/data-structure";
import * as content from "../../../content/content";
import * as metricsHelpersImpressions from "../../../metrics/helpers/impressions";
import * as metricsHelpersLocation from "../../../metrics/helpers/location";
import * as editorialActionUtil from "../../editorial-action-util";
import * as editorialDataUtil from "../../editorial-data-util";
import * as editorialMediaUtil from "../../editorial-page-component-media-util";
import * as editorialCopyUtil from "../../editorial-page-editorial-copy-util";
import { CollectionShelfDisplayStyle } from "../../editorial-page-types";
import { createShelfTokenUrlIfNecessary, shouldDefer } from "./editorial-page-collection-shelf-common";
import { deepLinkUrlFromData } from "../../../linking/external-deep-link";
import { isMediaDark } from "../../editorial-media-util";
import { insertPlaceholdersIntoShelfIfRequired } from "../../../placeholders/placeholders";
import { convertToShelfModel, makeShelfId, } from "../../../../gameservicesui/src/editorial-page/editorial-component-builder";
import { createSmallBrickFromData } from "../../../../gameservicesui/src/models/cards/small-brick";
export function buildBrickSmallShelf(objectGraph, shelfToken) {
    return buildBrickShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.BrickSmall, "smallBrick");
}
export function buildBrickMediumShelf(objectGraph, shelfToken) {
    return buildBrickShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.BrickMedium, "brick");
}
export function buildBrickLargeShelf(objectGraph, shelfToken) {
    return buildBrickShelf(objectGraph, shelfToken, CollectionShelfDisplayStyle.BrickLarge, "largeBrick");
}
// region Brick Creation
function buildBrickShelf(objectGraph, shelfToken, displayStyle, brickComponentType) {
    const items = [];
    const shelf = new models.Shelf(brickComponentType);
    shelf.isHorizontal = mediaAttributes.attributeAsString(shelfToken.data, "layoutDirection") === "Horizontal";
    const shelfContents = editorialDataUtil.extractRemainingItemsOrInitialShelfContents(objectGraph, shelfToken);
    for (const itemData of shelfContents) {
        const metricsOptions = {
            ...shelfToken.metricsImpressionOptions,
            targetType: editorialActionUtil.clickTargetForCollectionDisplayStyle(objectGraph, displayStyle),
            recoMetricsData: mediaDataStructure.metricsFromMediaApiObject(itemData),
        };
        if (!itemData.attributes || shouldDefer(shelfToken)) {
            shelfToken.remainingItems.push(itemData);
            shelfToken.isDeferring = true;
            continue;
        }
        if (preprocessor.GAMES_TARGET && displayStyle === CollectionShelfDisplayStyle.BrickSmall) {
            const clickAction = editorialActionUtil.createPrimaryActionForComponentFromData(objectGraph, itemData, shelfToken);
            const smallBrick = createSmallBrickFromData(objectGraph, itemData, makeShelfId(shelfToken), clickAction, {}, {}, true);
            if (isNothing(smallBrick)) {
                continue;
            }
            items.push(convertToShelfModel(smallBrick));
        }
        else {
            const brick = buildBrick(objectGraph, itemData, shelfToken.collectionDisplayStyle, metricsOptions, metricsOptions, shelfToken.clientIdentifierOverride);
            brick.clickAction = editorialActionUtil.createPrimaryActionForComponentFromData(objectGraph, itemData, shelfToken);
            if (!brick.isValid()) {
                continue;
            }
            items.push(brick);
        }
        metricsHelpersLocation.nextPosition(shelfToken.metricsImpressionOptions.locationTracker);
    }
    if (serverData.isDefinedNonNull(shelfToken.presentationHints)) {
        shelf.presentationHints = shelfToken.presentationHints;
    }
    // We don't need this in our incomplete shelf URL, so we'll preemptively remove it.
    delete shelfToken.maxItemCount;
    shelf.eyebrow = shelfToken.eyebrow;
    shelf.eyebrowArtwork = shelfToken.eyebrowArtwork;
    shelf.title = shelfToken.title;
    shelf.subtitle = shelfToken.subtitle;
    shelf.items = items;
    shelf.rowsPerColumn = mediaAttributes.attributeAsNumber(shelfToken.data, "rowCount");
    shelf.url = createShelfTokenUrlIfNecessary(objectGraph, shelf, shelfToken);
    insertPlaceholdersIntoShelfIfRequired(objectGraph, shelf, shelfToken);
    return shelf;
}
/**
 * Creates a Brick from the input data.
 * Note: the clickAction will need to be assigned post-creation.
 *
 * @param objectGraph Current object graph
 * @param data The data for the Brick
 * @param collectionDisplayStyle The desired display style for the brick
 * @param metricsOptions Metrics options for impressions
 * @param lockupMetricsOptions Metrics options for the collection icon lockups
 * @param clientIdentifierOverride Any preferred client identifier override
 * @returns A built Brick object
 */
export function buildBrick(objectGraph, data, collectionDisplayStyle, metricsOptions, lockupMetricsOptions, clientIdentifierOverride) {
    const brick = new models.Brick();
    const editorialClientParams = editorialDataUtil.extractEditorialClientParams(objectGraph, data);
    const hideEditorialMedia = serverData.asBooleanOrFalse(editorialClientParams.hideEditorialMedia);
    if (!hideEditorialMedia) {
        const editorialMediaData = editorialMediaUtil.editorialMediaDataFromData(objectGraph, data, collectionDisplayStyle);
        if (isSome(editorialMediaData === null || editorialMediaData === void 0 ? void 0 : editorialMediaData.artwork)) {
            brick.artworks = [editorialMediaData.artwork];
            brick.isMediaDark = isMediaDark(objectGraph, editorialMediaData);
        }
        if (isSome(editorialMediaData === null || editorialMediaData === void 0 ? void 0 : editorialMediaData.rtlArtwork)) {
            brick.rtlArtwork = editorialMediaData === null || editorialMediaData === void 0 ? void 0 : editorialMediaData.rtlArtwork;
            brick.isRTLMediaDark = isMediaDark(objectGraph, editorialMediaData, true);
        }
    }
    const lockupOptions = {
        metricsOptions: lockupMetricsOptions,
        clientIdentifierOverride: clientIdentifierOverride,
        artworkUseCase: 0 /* content.ArtworkUseCase.Default */,
        canDisplayArcadeOfferButton: true,
        externalDeepLinkUrl: deepLinkUrlFromData(objectGraph, data),
    };
    metricsHelpersLocation.pushBasicLocation(objectGraph, lockupOptions.metricsOptions, "collectionIcons");
    brick.collectionIcons = editorialMediaUtil.editorialFallbackAppIconsFromData(objectGraph, data, lockupOptions);
    metricsHelpersLocation.popLocation(lockupOptions.metricsOptions.locationTracker);
    if (serverData.isNullOrEmpty(brick.collectionIcons)) {
        const primaryAppIcon = content.iconFromData(objectGraph, data, {
            useCase: 0 /* content.ArtworkUseCase.Default */,
        });
        if (serverData.isDefinedNonNullNonEmpty(primaryAppIcon)) {
            brick.collectionIcons = [primaryAppIcon];
        }
    }
    brick.caption = editorialCopyUtil.editorialBadgeFromData(objectGraph, data, collectionDisplayStyle);
    brick.title = editorialCopyUtil.editorialTitleFromData(objectGraph, data, collectionDisplayStyle);
    brick.subtitle = editorialCopyUtil.editorialDescriptionFromData(objectGraph, data);
    brick.accessibilityLabel = brick.title;
    brick.editorialDisplayOptions = editorialDataUtil.editorialDisplayOptionsFromClientParams(editorialClientParams);
    // Configure impressions
    const metricsIdType = serverData.isDefinedNonNullNonEmpty(brick.artworks)
        ? "its_contentId"
        : "collection_id";
    const contentMetricsOptions = {
        ...metricsOptions,
        id: data.id,
        idType: metricsIdType,
    };
    const impressionOptions = metricsHelpersImpressions.impressionOptions(objectGraph, data, brick.title, contentMetricsOptions);
    metricsHelpersImpressions.addImpressionFields(objectGraph, brick, impressionOptions);
    // Safe area
    brick.artworkSafeArea = models.ChartOrCategorySafeArea.defaultTileArtworkSafeArea;
    brick.textSafeArea = models.ChartOrCategorySafeArea.defaultTileTextSafeArea;
    return brick;
}
// endregion
//# sourceMappingURL=editorial-page-brick-collection-shelf-builder.js.map