import * as models from "../../../../api/models";
import * as serverData from "../../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../../foundation/media/attributes";
import * as mediaDataStructure from "../../../../foundation/media/data-structure";
import * as metricsHelpersImpressions from "../../../metrics/helpers/impressions";
import * as editorialActionUtil from "../../editorial-action-util";
import * as editorialDataUtil from "../../editorial-data-util";
import * as editorialComponentMediaUtil from "../../editorial-page-component-media-util";
import * as editorialCopyUtil from "../../editorial-page-editorial-copy-util";
import * as editorialOverlayContentUtil from "../../editorial-page-overlay-content-util";
import { createShelfTokenUrlIfNecessary, shouldDefer } from "./editorial-page-collection-shelf-common";
import { deepLinkUrlFromData } from "../../../linking/external-deep-link";
import { isMediaDark } from "../../editorial-media-util";
import { insertPlaceholdersIntoShelfIfRequired } from "../../../placeholders/placeholders";
export function buildBreakoutLargeShelf(objectGraph, shelfToken) {
    const items = [];
    const shelf = new models.Shelf("largeHeroBreakout");
    shelf.isHorizontal = mediaAttributes.attributeAsString(shelfToken.data, "layoutDirection") === "Horizontal";
    const shelfContents = editorialDataUtil.extractRemainingItemsOrInitialShelfContents(objectGraph, shelfToken);
    for (const itemData of shelfContents) {
        const primaryItemData = editorialDataUtil.extractHydratedPrimaryContentData(objectGraph, itemData);
        if ((editorialDataUtil.requiresPrimaryContent(objectGraph, itemData) &&
            !mediaAttributes.hasAttributes(primaryItemData)) ||
            shouldDefer(shelfToken)) {
            shelfToken.isDeferring = true;
            shelfToken.remainingItems.push(itemData);
            continue;
        }
        const metricsOptions = {
            targetType: "largeBreakout",
            pageInformation: shelfToken.metricsPageInformation,
            locationTracker: shelfToken.metricsLocationTracker,
            recoMetricsData: mediaDataStructure.metricsFromMediaApiObject(itemData),
        };
        const editorialClientParams = editorialDataUtil.extractEditorialClientParams(objectGraph, itemData);
        const hideEditorialMedia = serverData.asBooleanOrFalse(editorialClientParams.hideEditorialMedia);
        const mediaData = !hideEditorialMedia
            ? editorialComponentMediaUtil.editorialMediaDataFromData(objectGraph, itemData, shelfToken.collectionDisplayStyle)
            : null;
        const lockupOptions = {
            metricsOptions,
            clientIdentifierOverride: shelfToken.clientIdentifierOverride,
            artworkUseCase: 0 /* content.ArtworkUseCase.Default */,
            offerEnvironment: "lightOverArtwork",
            canDisplayArcadeOfferButton: true,
            externalDeepLinkUrl: deepLinkUrlFromData(objectGraph, itemData),
        };
        const overlayContent = editorialOverlayContentUtil.extractOverlayContent(objectGraph, itemData, lockupOptions);
        if (serverData.isNullOrEmpty(mediaData) && serverData.isNullOrEmpty(overlayContent.collectionIcons)) {
            continue;
        }
        const badgeText = editorialCopyUtil.editorialBadgeFromData(objectGraph, itemData, shelfToken.collectionDisplayStyle);
        const badgeType = serverData.isDefinedNonNull(badgeText)
            ? { type: "text", title: badgeText }
            : { type: "none", title: null };
        const largeBreakoutDetails = new models.BreakoutDetails(editorialCopyUtil.editorialTitleFromData(objectGraph, itemData, shelfToken.collectionDisplayStyle), editorialCopyUtil.editorialDescriptionFromData(objectGraph, itemData, true), badgeType, null);
        const largeBreakout = new models.LargeHeroBreakout(largeBreakoutDetails, { position: "leading", wantsBlur: true }, null, mediaData === null || mediaData === void 0 ? void 0 : mediaData.artwork, mediaData === null || mediaData === void 0 ? void 0 : mediaData.video, overlayContent.collectionIcons, mediaData === null || mediaData === void 0 ? void 0 : mediaData.backgroundColor);
        largeBreakout.rtlArtwork = mediaData === null || mediaData === void 0 ? void 0 : mediaData.rtlArtwork;
        largeBreakout.rtlVideo = mediaData === null || mediaData === void 0 ? void 0 : mediaData.rtlVideo;
        largeBreakout.rtlBackgroundColor = mediaData === null || mediaData === void 0 ? void 0 : mediaData.rtlBackgroundColor;
        // Configure impressions
        const contentMetricsOptions = {
            ...metricsOptions,
            id: itemData.id,
            idType: "its_contentId",
        };
        const impressionOptions = metricsHelpersImpressions.impressionOptions(objectGraph, itemData, largeBreakout.details.title, contentMetricsOptions);
        metricsHelpersImpressions.addImpressionFields(objectGraph, largeBreakout, impressionOptions);
        const breakoutAction = editorialActionUtil.createPrimaryActionForComponentFromData(objectGraph, itemData, shelfToken);
        if (serverData.isDefinedNonNull(breakoutAction)) {
            breakoutAction.title = editorialCopyUtil.editorialCallToActionFromData(objectGraph, itemData);
        }
        largeBreakout.details.callToActionButtonAction = breakoutAction;
        largeBreakout.clickAction = breakoutAction;
        largeBreakout.editorialDisplayOptions =
            editorialDataUtil.editorialDisplayOptionsFromClientParams(editorialClientParams);
        largeBreakout.lockup = overlayContent.lockup;
        largeBreakout.isMediaDark = isMediaDark(objectGraph, mediaData);
        largeBreakout.isRTLMediaDark = isMediaDark(objectGraph, mediaData, true);
        items.push(largeBreakout);
    }
    // We don't need this in our incomplete shelf URL, so we'll preemptively remove it.
    delete shelfToken.maxItemCount;
    shelf.eyebrow = shelfToken.eyebrow;
    shelf.eyebrowArtwork = shelfToken.eyebrowArtwork;
    shelf.title = shelfToken.title;
    shelf.subtitle = shelfToken.subtitle;
    shelf.items = items;
    shelf.rowsPerColumn = mediaAttributes.attributeAsNumber(shelfToken.data, "rowCount");
    shelf.url = createShelfTokenUrlIfNecessary(objectGraph, shelf, shelfToken);
    insertPlaceholdersIntoShelfIfRequired(objectGraph, shelf, shelfToken);
    return shelf;
}
//# sourceMappingURL=editorial-page-breakout-large-collection-shelf-builder.js.map