import * as serverData from "../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../foundation/media/attributes";
import * as mediaPlatformAttributes from "../../foundation/media/platform-attributes";
import * as contentAttributes from "../content/attributes";
import * as editorialDataUtil from "./editorial-data-util";
/**
 * Get the text that should be displayed as the badge (some times referred to as caption) for
 * an editorial component
 * @param objectGraph
 * @param data
 * @param collectionDisplayStyle
 */
export function editorialBadgeFromData(objectGraph, data, collectionDisplayStyle) {
    let editorialBadge = null;
    let editorialBadgeOverride = null;
    let editorialBadgePaths = [];
    // Check the editorial card first for override content
    const editorialCardData = editorialDataUtil.extractEditorialCardData(objectGraph, data);
    switch (data.type) {
        case "editorial-items":
        case "editorial-pages":
        case "editorial-shelves-header":
            editorialBadgePaths = ["enrichedEditorialNotes.badge", "editorialNotes.badge", "label"];
            editorialBadge = findFirstMatchingString(objectGraph, data, editorialBadgePaths, false);
            editorialBadgeOverride = findFirstMatchingString(objectGraph, editorialCardData, editorialBadgePaths, false);
            break;
        case "apps":
        case "in-apps":
        case "app-bundles":
            editorialBadgePaths = ["editorialNotes.badge"];
            editorialBadge = findFirstMatchingString(objectGraph, data, editorialBadgePaths, true);
            editorialBadgeOverride = findFirstMatchingString(objectGraph, editorialCardData, editorialBadgePaths, true);
            break;
        default:
            break;
    }
    const badge = serverData.isDefinedNonNull(editorialBadgeOverride) ? editorialBadgeOverride : editorialBadge;
    if (serverData.isNullOrEmpty(badge)) {
        return null;
    }
    return badge;
}
/**
 * Get the text that should be displayed as the title for
 * an editorial component
 * @param objectGraph
 * @param data
 * @param collectionDisplayStyle
 */
export function editorialTitleFromData(objectGraph, data, collectionDisplayStyle) {
    let editorialTitle = null;
    let editorialTitlePaths = [];
    switch (collectionDisplayStyle) {
        case "EditorialLockupMediumVariant":
        case "EditorialLockupLargeVariant":
            switch (data.type) {
                case "editorial-items":
                case "editorial-pages":
                case "editorial-shelves-header":
                    editorialTitlePaths = ["enrichedEditorialNotes.name", "editorialNotes.name"];
                    editorialTitle = attributeAsString(data, editorialTitlePaths);
                    break;
                case "apps":
                case "in-apps":
                case "app-bundles":
                    editorialTitlePaths = ["editorialNotes.tagline"];
                    editorialTitle = findFirstMatchingString(objectGraph, data, editorialTitlePaths, true);
                    break;
                default:
                    break;
            }
            break;
        default:
            switch (data.type) {
                case "editorial-items":
                case "editorial-pages":
                case "editorial-shelves-header":
                    editorialTitlePaths = ["enrichedEditorialNotes.name", "editorialNotes.name"];
                    editorialTitle = findFirstMatchingString(objectGraph, data, editorialTitlePaths, false);
                    break;
                case "tags":
                case "apps":
                case "in-apps":
                case "app-bundles":
                    editorialTitlePaths = ["name"];
                    editorialTitle = findFirstMatchingString(objectGraph, data, editorialTitlePaths, false);
                    break;
                default:
                    break;
            }
    }
    // Check the editorial card first for override content
    const editorialCardData = editorialDataUtil.extractEditorialCardData(objectGraph, data);
    const editorialTitleOverride = findFirstMatchingString(objectGraph, editorialCardData, editorialTitlePaths, false);
    const title = serverData.isDefinedNonNull(editorialTitleOverride) ? editorialTitleOverride : editorialTitle;
    if (serverData.isNullOrEmpty(title)) {
        return null;
    }
    return title;
}
/**
 * Get the text that should be displayed as the description for
 * an editorial component
 * @param objectGraph
 * @param data
 */
export function editorialDescriptionFromData(objectGraph, data, preferHeroDescription = false) {
    let editorialDescription = null;
    let editorialDescriptionPaths = [];
    switch (data.type) {
        case "editorial-items":
            if (preferHeroDescription) {
                editorialDescriptionPaths = ["enrichedEditorialNotes.tagline", "editorialNotes.tagline"];
            }
            else {
                const ignoreShortNotes = mediaAttributes.attributeAsBooleanOrFalse(data, "ignoreITunesShortNotes");
                if (!ignoreShortNotes) {
                    editorialDescriptionPaths = ["enrichedEditorialNotes.short", "editorialNotes.short"];
                }
            }
            editorialDescription = findFirstMatchingString(objectGraph, data, editorialDescriptionPaths, false);
            break;
        case "editorial-pages":
        case "editorial-shelves-header":
            editorialDescriptionPaths = ["enrichedEditorialNotes.tagline", "editorialNotes.tagline"];
            editorialDescription = findFirstMatchingString(objectGraph, data, editorialDescriptionPaths, false);
            break;
        case "apps":
        case "in-apps":
        case "app-bundles":
            editorialDescriptionPaths = ["editorialNotes.tagline"];
            editorialDescription = findFirstMatchingString(objectGraph, data, editorialDescriptionPaths, true);
            break;
        default:
            break;
    }
    // Check the editorial card first for override content
    const editorialCardData = editorialDataUtil.extractEditorialCardData(objectGraph, data);
    const editorialDescriptionOverride = findFirstMatchingString(objectGraph, editorialCardData, editorialDescriptionPaths, false);
    const description = serverData.isDefinedNonNull(editorialDescriptionOverride)
        ? editorialDescriptionOverride
        : editorialDescription;
    if (serverData.isNullOrEmpty(description)) {
        return null;
    }
    return description;
}
/**
 * Get the text that should be displayed as the call to action on a button for
 * an editorial component
 * @param objectGraph
 * @param data
 */
export function editorialCallToActionFromData(objectGraph, data) {
    let editorialCallToAction = null;
    let editorialCallToActionPaths = [];
    switch (data.type) {
        case "editorial-items":
        case "editorial-pages":
        case "editorial-shelves-header":
            editorialCallToActionPaths = ["enrichedEditorialNotes.callToAction", "editorialNotes.callToAction"];
            editorialCallToAction = findFirstMatchingString(objectGraph, data, editorialCallToActionPaths, false);
            break;
        case "apps":
        case "in-apps":
        case "app-bundles":
            editorialCallToActionPaths = ["enrichedEditorialNotes.callToAction", "editorialNotes.callToAction"];
            editorialCallToAction = findFirstMatchingString(objectGraph, data, editorialCallToActionPaths, true);
            break;
        default:
            break;
    }
    // Check the editorial card first for override content
    const editorialCardData = editorialDataUtil.extractEditorialCardData(objectGraph, data);
    const editorialCallToActionOverride = findFirstMatchingString(objectGraph, editorialCardData, editorialCallToActionPaths, false);
    let callToAction = null;
    if ((editorialCallToActionOverride === null || editorialCallToActionOverride === void 0 ? void 0 : editorialCallToActionOverride.length) > 0) {
        callToAction = editorialCallToActionOverride;
    }
    else if (serverData.isDefinedNonNullNonEmpty(editorialCallToAction)) {
        callToAction = editorialCallToAction;
    }
    else {
        if (preprocessor.GAMES_TARGET) {
            callToAction = objectGraph.loc.string("Card.Collection.ActionButton.Title");
        }
        else {
            callToAction = objectGraph.loc.string("HERO_CAROUSEL_OVERLAY_BUTTON_TITLE_VIEW");
        }
    }
    return callToAction;
}
// region Debug Helpers
/**
 * Retrieve the specified attribute from the data as a string.
 *
 * @param data The data from which to retrieve the attribute.
 * @param attributePath The object path for the attribute.
 * @returns {string} The attribute value as a string.
 */
function attributeAsString(data, attributePath) {
    if (serverData.isNullOrEmpty(data)) {
        return null;
    }
    return mediaAttributes.attributeAsString(data, attributePath);
}
/**
 * Retrieve the specified attribute from the data as a string. Returning the path if null or missing
 *
 * If the attribute lives under the platform-specific attributes, then a platform may be provided to properly call in to
 * the nested structure.
 * @param data The data from which to retrieve the attribute.
 * @param platform The platform to look up
 * @param attributePath The object path for the attribute.
 * @returns {string} The attribute value as a string.
 */
export function platformAttributeAsString(data, platform, attributePath) {
    if (serverData.isNullOrEmpty(data)) {
        return null;
    }
    return mediaPlatformAttributes.platformAttributeAsString(data, platform, attributePath);
}
/**
 * Retrieve the specified attribute from the data as a string. Trying to find the best match for the given paths.
 * @param objectGraph The object graph to use lookups
 * @param mediaApiData The media data to use to query for the string
 * @param stringPaths The different paths to try to find the string
 * @param usePlatformAttributes Whether to check the platform attributes or not
 * @returns The first matching string or null if none found
 */
function findFirstMatchingString(objectGraph, mediaApiData, stringPaths, usePlatformAttributes) {
    if (serverData.isNullOrEmpty(mediaApiData)) {
        return null;
    }
    const attributePlatform = contentAttributes.bestAttributePlatformFromData(objectGraph, mediaApiData);
    for (const path of stringPaths) {
        const matchingString = usePlatformAttributes
            ? platformAttributeAsString(mediaApiData, attributePlatform, path)
            : attributeAsString(mediaApiData, path);
        if (serverData.isDefinedNonNullNonEmpty(matchingString)) {
            return matchingString;
        }
    }
    return null;
}
// endregion
//# sourceMappingURL=editorial-page-editorial-copy-util.js.map