import * as serverData from "../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../foundation/media/attributes";
import * as mediaRelationship from "../../foundation/media/relationships";
import * as content from "../content/content";
import { EditorialShelfType } from "./editorial-page-types";
/**
 * Take the remaining content fromt he shelfToken if this is not the first render, otherwise
 * use the initial data.
 *
 * @param objectGraph
 * @param shelfToken
 */
export function extractRemainingItemsOrInitialShelfContents(objectGraph, shelfToken) {
    if (!shelfToken.isFirstRender) {
        return serverData.isDefinedNonNullNonEmpty(shelfToken.remainingItems) ? shelfToken.remainingItems : [];
    }
    const initialShelfContent = mediaRelationship.relationshipCollection(shelfToken.data, "contents");
    return serverData.isDefinedNonNullNonEmpty(initialShelfContent) ? initialShelfContent : [];
}
/**
 * Find the related product data from some MAPI data that was programmed on an editorial page.
 *
 * @param objectGraph
 * @param data
 * @returns the product MAPI data from the correct relationship or the data itself if the supplied data is already a product
 */
export function extractProductData(objectGraph, data) {
    let productData = null;
    switch (data.type) {
        case "editorial-items":
        case "editorial-pages":
        case "editorial-shelves-header":
            productData =
                mediaRelationship.relationshipData(objectGraph, data, "primary-content") ||
                    mediaRelationship.relationshipData(objectGraph, data, "card-contents") ||
                    mediaRelationship.relationshipData(objectGraph, data, "contents") ||
                    mediaRelationship.relationshipData(objectGraph, data, "app");
            break;
        case "apps":
        case "in-apps":
        case "app-bundles":
            productData = data;
            break;
        default:
            break;
    }
    return productData;
}
/**
 * Find the related collection of product data from some MAPI data that was programmed on an editorial page.
 *
 * @param objectGraph
 * @param data
 * @returns collection of related content for in the relationship of the supplied MAPI data
 */
export function extractCollectionData(objectGraph, data) {
    let collectionData = null;
    switch (data.type) {
        case "editorial-items":
        case "editorial-pages":
        case "editorial-shelves-header":
            collectionData =
                mediaRelationship.relationshipCollection(data, "primary-contents", true) ||
                    mediaRelationship.relationshipCollection(data, "card-contents", true) ||
                    mediaRelationship.relationshipCollection(data, "contents", true);
            break;
        case "tags":
            collectionData = serverData.asArray(data.meta, "associations.apps.data");
            break;
        default:
            collectionData = null;
    }
    return collectionData;
}
/**
 * Find the related editorial card used for determining override information
 *
 * @param objectGraph
 * @param data
 * @returns The editorial-card data object to use for overrides
 */
export function extractEditorialCardData(objectGraph, data) {
    if (serverData.isNullOrEmpty(data)) {
        return null;
    }
    const editorialCardsData = serverData.asArrayOrEmpty(data.meta, "associations.editorial-cards.data");
    if (serverData.isNullOrEmpty(editorialCardsData)) {
        return null;
    }
    return editorialCardsData[0];
}
/**
 * Find the related editorial client params from this data object
 *
 * @param objectGraph
 * @param data
 * @returns The EditorialClientParams data object to use for this component
 */
export function extractEditorialClientParams(objectGraph, data) {
    const baseEditorialClientParams = mediaAttributes.attributeAsDictionary(data, "editorialClientParams", {});
    const editorialCardData = extractEditorialCardData(objectGraph, data);
    const cardEditorialClientParams = mediaAttributes.attributeAsDictionary(editorialCardData, "editorialClientParams", {});
    return serverData.asInterface({
        ...baseEditorialClientParams,
        ...cardEditorialClientParams,
    });
}
/**
 * Use the client params to create the display options for this component
 *
 * @param clientParams
 * @returns The editorialDisplayOptions data object to use for this component
 */
export function editorialDisplayOptionsFromClientParams(clientParams) {
    return {
        showAppIcon: serverData.asBooleanWithDefault(clientParams.showAppIcon, false),
        suppressLockup: serverData.asBooleanWithDefault(clientParams.suppressLockup, false),
        useGeneratedBackground: serverData.asBooleanWithDefault(clientParams.useGeneratedBackground, false),
        useMaterialBlur: serverData.asBooleanWithDefault(clientParams.useMaterialBlur, true),
        useTextProtectionColor: serverData.asBooleanWithDefault(clientParams.useTextProtectionColor, false),
    };
}
/**
 * Get the primary content for this MAPI data object, depending on the type it is. If the data
 * does not have any attributes this method will return null, since we'll need to fetch it.
 * @param objectGraph
 * @param data
 */
export function extractHydratedPrimaryContentData(objectGraph, data) {
    if (!mediaAttributes.hasAttributes(data)) {
        return null;
    }
    let primaryData;
    switch (data.type) {
        case "editorial-items":
            primaryData = content.primaryContentForData(objectGraph, data);
            break;
        default:
            primaryData = data;
            break;
    }
    if (!mediaAttributes.hasAttributes(primaryData)) {
        return null;
    }
    return primaryData;
}
/**
 * Method to check whether we need the primary content relationship for a piece of content
 * to be rendered
 *
 * @param objectGraph
 * @param data The MAPI date we're checking for
 */
export function requiresPrimaryContent(objectGraph, data) {
    const linkData = mediaAttributes.attributeAsDictionary(data, "link");
    const isLinkAction = serverData.isDefinedNonNullNonEmpty(linkData);
    const isStoryAction = mediaAttributes.attributeAsBooleanOrFalse(data, "isCanvasAvailable");
    return !isLinkAction && !isStoryAction;
}
/**
 * Extracts the collection shelf display style, for any collection / recommendation shelf
 * @param objectGraph Current object graph
 * @param shelfData The input shelf data
 * @returns The collection shelf display style, if any
 */
export function collectionShelfDisplayStyleFromShelfData(objectGraph, shelfData) {
    const shelfType = shelfData.type;
    if (shelfType !== EditorialShelfType.Collection && shelfType !== EditorialShelfType.Recommendations) {
        return null;
    }
    return mediaAttributes.attributeAsString(shelfData, "displayStyle");
}
//# sourceMappingURL=editorial-data-util.js.map