import { isSome } from "@jet/environment/types/optional";
import * as serverData from "../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../foundation/media/attributes";
import * as contentAttributes from "./attributes";
let implementation = null;
/**
 * Creates an implementation of the `SystemApps` interface
 * which is appropriate for the current platform.
 */
export function systemApps(objectGraph) {
    if (implementation !== null) {
        return implementation;
    }
    // Because watchOS is not a proper platform from the back end's perspective,
    // we must map adam IDs to bundle IDs using a bag property.
    if (objectGraph.client.isWatch) {
        const systemAppsMap = new Map();
        const nonDeletableSystemAppsMap = new Map();
        for (const systemApp of objectGraph.bag.systemApps) {
            const bundleId = serverData.asString(systemApp, "bundle-id", "coercible");
            const adamId = serverData.asString(systemApp, "id", "coercible");
            if (isSome(bundleId) && isSome(adamId)) {
                systemAppsMap.set(adamId, bundleId);
            }
        }
        for (const nonDeletableSystemApp of objectGraph.bag.nonDeletableSystemApps) {
            const bundleId = serverData.asString(nonDeletableSystemApp, "bundle-id", "coercible");
            const adamId = serverData.asString(nonDeletableSystemApp, "id", "coercible");
            if (isSome(bundleId) && isSome(adamId)) {
                nonDeletableSystemAppsMap.set(adamId, bundleId);
            }
        }
        const unsupportedSystemApps = new Map();
        implementation = {
            bundleIdFromData(data) {
                const mappedBundleId = systemAppsMap.get(data.id);
                if (!serverData.isNull(mappedBundleId)) {
                    return mappedBundleId;
                }
                else {
                    return contentAttributes.contentAttributeAsString(objectGraph, data, "bundleId");
                }
            },
            isSystemAppFromData(data) {
                return systemAppsMap.has(data.id);
            },
            isUnsupportedDeletableSystemAppFromData(data) {
                if (systemAppsMap.has(data.id) && !nonDeletableSystemAppsMap.has(data.id)) {
                    // We must use the bundleID from the bag here, rather than a bundleID from `data`,
                    // as AppConduit is expecting the bundleIDs which are provided in the bag
                    const bundleId = systemAppsMap.get(data.id);
                    if (serverData.isDefinedNonNullNonEmpty(bundleId)) {
                        const valueFromMap = unsupportedSystemApps.get(bundleId);
                        if (isSome(valueFromMap)) {
                            return valueFromMap;
                        }
                        const isUnsupported = !objectGraph.client.deletableSystemAppCanBeInstalledOnWatchWithBundleID(bundleId);
                        unsupportedSystemApps.set(bundleId, isUnsupported);
                        return isUnsupported;
                    }
                }
                return false;
            },
            adamIdFromSystemBundleId(bundleId) {
                for (const [key, value] of systemAppsMap) {
                    if (value === bundleId) {
                        return key;
                    }
                }
                return undefined;
            },
        };
    }
    else {
        implementation = {
            bundleIdFromData(data) {
                return contentAttributes.contentAttributeAsString(objectGraph, data, "bundleId");
            },
            isSystemAppFromData(data) {
                return mediaAttributes.attributeAsBooleanOrFalse(data, "isFirstPartyHideableApp");
            },
            isUnsupportedDeletableSystemAppFromData(data) {
                return false;
            },
            adamIdFromSystemBundleId(bundleId) {
                // This path shouldn't run on any other platforms.
                return undefined;
            },
        };
    }
    return implementation;
}
//# sourceMappingURL=sad.js.map