import { isSome } from "@jet/environment";
import * as mediaAttributes from "../../foundation/media/attributes";
import * as serverData from "../../foundation/json-parsing/server-data";
/**
 * Provides the localized name of the product's age rating, e.g. "12+".
 * @param objectGraph The App Store object graph.
 * @param data The Media API data for an app.
 * @param useLegacyFallback Indicates whether the legacy key should be used if the modern one is missing.
 * @returns The localized `string` of the product's age rating, or `undefined`.
 */
export function name(objectGraph, data, useLegacyFallback = false) {
    const modernValue = mediaAttributes.attributeAsString(data, "ageRating.name");
    if (isSome(modernValue)) {
        return modernValue;
    }
    else if (useLegacyFallback) {
        return mediaAttributes.attributeAsString(data, "contentRatingsBySystem.appsApple.name");
    }
    else {
        return undefined;
    }
}
/**
 * Provides the product's age rating value, e.g. 300. This value is understood
 * by systems such as Managed Configuration and Screen Time to enforce content
 * restrictions.
 * @param objectGraph The App Store object graph.
 * @param data The Media API data for an app.
 * @param useLegacyFallback Indicates whether the legacy key should be used if the modern one is missing.
 * @returns The `number` value of the product's age rating, or `undefined`.
 */
export function value(objectGraph, data, useLegacyFallback = false) {
    const modernValue = mediaAttributes.attributeAsNumber(data, "ageRating.value");
    if (isSome(modernValue)) {
        return modernValue;
    }
    else if (useLegacyFallback) {
        return mediaAttributes.attributeAsNumber(data, "contentRatingsBySystem.appsApple.value");
    }
    else {
        return undefined;
    }
}
/**
 * Provides the product's age rating description from Media API. This was
 * historically generated by the client, so has no legacy fallback.
 * @param objectGraph The App Store object graph.
 * @param data The Media API data for an app.
 * @returns The product's age rating description `string`, or `undefined`.
 */
export function description(objectGraph, data) {
    return mediaAttributes.attributeAsString(data, "ageRating.description");
}
export function hasInAppControls(objectGraph, data) {
    const contentLevels = mediaAttributes.attributeAsArrayOrEmpty(data, "ageRating.contentLevels");
    for (const contentLevel of contentLevels) {
        if (serverData.asString(contentLevel, "kind") === "IAC") {
            return true;
        }
    }
    return false;
}
/**
 * Provides the product's developer age guidance URL from Media API. This is an
 * optional URL the developer can provide in App Store Connect to give further
 * details on their app's content controls. There is no legacy fallback for
 * this value.
 * @param objectGraph The App Store object graph.
 * @param data The Media API data for an app.
 * @returns The product's developer age guidance URL `string`, or `undefined`.
 */
export function developerAgeGuidanceURL(objectGraph, data) {
    return mediaAttributes.attributeAsString(data, "ageRating.ageGuidanceUrl");
}
/**
 * Returns the name of an image resource in the App Store bundle corresponding
 * to the provided MAPI `ageRating` data.
 * @param objectGraph The App Store object graph.
 * @param data The Media API data for an app.
 * @returns The `string` name of an image resource in the bundle, or `undefined`.
 */
export function pictogramResource(objectGraph, data) {
    // Values pulled from https://quip-apple.com/0bq3AiLxhzaW
    const pictogramResources = new Map([
        // Brazil
        ["br.100.official", "br.l.official"],
        ["br.100", "br.l"],
        ["br.210.official", "br.10.official"],
        ["br.210", "br.10"],
        ["br.300.official", "br.12.official"],
        ["br.300", "br.12"],
        ["br.314.official", "br.14.official"],
        ["br.314", "br.14"],
        ["br.416.official", "br.16.official"],
        ["br.416", "br.16"],
        ["br.618.official", "br.18.official"],
        ["br.618", "br.18"],
    ]);
    const storefront = objectGraph.locale.activeStorefront;
    const contentLevel = mediaAttributes.attributeAsString(data, "ageRating.value");
    const isOfficial = mediaAttributes.attributeAsBooleanOrFalse(data, "ageRating.isOfficial");
    const key = storefront + "." + contentLevel + (isOfficial ? ".official" : "");
    return pictogramResources.get(key);
}
//# sourceMappingURL=age-ratings.js.map