import * as models from "../../api/models";
import * as serverData from "../../foundation/json-parsing/server-data";
import * as color from "../../foundation/util/color-util";
import * as objects from "../../foundation/util/objects";
import * as metricsBuilder from "../metrics/builder";
import * as metricsHelpersLocation from "../metrics/helpers/location";
import * as metricsHelpersPage from "../metrics/helpers/page";
import { MetricsReferralContext } from "../metrics/metrics-referral-context";
import * as appPromotionCommon from "./app-promotions-common";
/**
 * Create a flow action for navigating to the contingent offer detail page.
 * @param data The data blob
 * @param parentAppData The associated parent app data
 * @param contingentOffer The source contingent offer
 * @param baseMetricsOptions The base metrics options
 * @param animationBehavior The animation behavior for presenting the modal page
 * @param includeLockupClickAction Whether to generate a click action for the lockup
 * @param referrerData Referrer data from an incoming deep link
 */
export function contingentOfferDetailPageFlowActionFromData(objectGraph, data, parentAppData, contingentOffer, baseMetricsOptions, animationBehavior, includeLockupClickAction, referrerData) {
    const page = contingentOfferDetailPageFromData(objectGraph, data, parentAppData, contingentOffer, baseMetricsOptions, includeLockupClickAction, referrerData, false);
    const action = new models.FlowAction("contingentOfferDetail");
    action.title = contingentOffer.title;
    action.pageData = page;
    action.animationBehavior = animationBehavior;
    if (baseMetricsOptions && baseMetricsOptions.pageInformation) {
        action.referrerUrl = baseMetricsOptions.pageInformation.pageUrl;
    }
    return action;
}
/**
 * Creates an contingent offer detail page
 * @param objectGraph The object graph
 * @param data The data blob
 * @param parentAppData The data blob for the related parent app
 * @param contingentOffer The source contingent offer
 * @param baseMetricsOptions The base metrics options to use for the detail page
 * @param includeLockupClickAction Whether to generate a click action for the lockup
 * @param referrerData The referrer data
 * @param isArcadePage Whether or not this is presented on the Arcade page
 */
export function contingentOfferDetailPageFromData(objectGraph, data, parentAppData, contingentOffer, baseMetricsOptions, includeLockupClickAction, referrerData, isArcadePage) {
    var _a, _b;
    const artwork = appPromotionCommon.artworkFromPlatformData(objectGraph, data, "productArtwork");
    const copy = objects.shallowCopyOf(contingentOffer);
    let mediaOverlayStyle = "dark";
    let isArtworkDark = true;
    if (serverData.isDefinedNonNull(artwork)) {
        isArtworkDark = color.isDarkColor(artwork.backgroundColor);
        mediaOverlayStyle = isArtworkDark ? "dark" : "light";
    }
    const pageInformation = metricsHelpersPage.pageInformationForAppPromotionDetailPage(objectGraph, models.AppPromotionType.ContingentOffer, data.id, parentAppData.id, referrerData, (_a = baseMetricsOptions.recoMetricsData) !== null && _a !== void 0 ? _a : null);
    const metricsOptions = {
        ...baseMetricsOptions,
        pageInformation: pageInformation,
        locationTracker: metricsHelpersLocation.newLocationTracker(),
    };
    const offerEnvironment = isArtworkDark ? "dark" : "light";
    const lockup = appPromotionCommon.lockupFromData(objectGraph, data, parentAppData, (_b = copy.title) !== null && _b !== void 0 ? _b : undefined, offerEnvironment, "transparent", false, metricsOptions, includeLockupClickAction, referrerData, isArcadePage, false);
    if (serverData.isNull(lockup)) {
        return null;
    }
    copy.offerLockup = lockup;
    copy.trunkAppIcon = contingentOffer.trunkAppIcon;
    const page = new models.ContingentOfferDetailPage(copy, artwork !== null && artwork !== void 0 ? artwork : undefined, mediaOverlayStyle);
    page.backButtonActionMetrics.addMetricsData(createButtonActionMetricsForTarget(objectGraph, "back", pageInformation, metricsOptions.locationTracker));
    page.learnMoreActionMetrics.addMetricsData(createButtonActionMetricsForTarget(objectGraph, "LearnMore", pageInformation, metricsOptions.locationTracker, contingentOffer.learnMoreTitle));
    page.closeButtonActionMetrics.addMetricsData(createButtonActionMetricsForTarget(objectGraph, "close", pageInformation, metricsOptions.locationTracker));
    metricsHelpersPage.addMetricsEventsToPageWithInformation(objectGraph, page, pageInformation, (fields) => {
        if (serverData.isDefinedNonNullNonEmpty(referrerData)) {
            MetricsReferralContext.shared.addReferralContextToMetricsFieldsIfNecessary(fields);
        }
    });
    return page;
}
function createButtonActionMetricsForTarget(objectGraph, targetId, pageInformation, locationTracker, title) {
    let actionType;
    switch (targetId) {
        case "LearnMore":
            actionType = "navigate";
            break;
        case "back":
            actionType = "back";
            break;
        case "close":
            actionType = "dismiss";
            break;
        default:
            break;
    }
    const eventFields = {
        targetType: "button",
        actionType,
        targetId,
        idType: undefined,
        location: metricsHelpersLocation.createContentLocation(objectGraph, {
            pageInformation: pageInformation,
            locationTracker: locationTracker,
            targetType: "button",
            id: targetId,
        }, title !== null && title !== void 0 ? title : targetId),
    };
    const event = metricsBuilder.createMetricsClickData(objectGraph, targetId, "button", eventFields);
    return event;
}
//# sourceMappingURL=contingent-offer-detail.js.map