import { isNothing, isSome } from "@jet/environment";
import * as models from "../../api/models";
import * as serverData from "../../foundation/json-parsing/server-data";
import * as mediaDataStructure from "../../foundation/media/data-structure";
import * as mediaRelationship from "../../foundation/media/relationships";
import * as metricsHelpersImpressions from "../metrics/helpers/impressions";
import * as metricsHelpersLocation from "../metrics/helpers/location";
import * as refresh from "../refresh/page-refresh-controller";
import * as appPromotionsCommon from "./app-promotions-common";
/**
 * Builder for the app events shelf, on the product page.
 * @param data The container data
 * @param shelfMetrics The product page shelf metrics.
 * @return The built shelf or null
 */
export function appPromotionsShelfForProductPage(objectGraph, data, shelfMetrics, refreshController) {
    var _a;
    if (!appPromotionsCommon.appEventsAreEnabled(objectGraph)) {
        return null;
    }
    let eventAndOffersData = mediaRelationship.relationshipViewsCollection(data, "events-and-offers");
    if (isNothing(eventAndOffersData) || eventAndOffersData.length === 0) {
        eventAndOffersData = mediaRelationship.relationshipCollection(data, "app-events");
    }
    if (isNothing(eventAndOffersData) || eventAndOffersData.length === 0) {
        return null;
    }
    const recoMetricsData = mediaRelationship.relationshipViewsContainer(data, "events-and-offers");
    const metricsOptions = {
        pageInformation: shelfMetrics.metricsPageInformation,
        locationTracker: shelfMetrics.locationTracker,
        recoMetricsData: (_a = mediaDataStructure.metricsFromMediaApiObject(recoMetricsData)) !== null && _a !== void 0 ? _a : undefined,
    };
    const hasOffers = eventAndOffersData.some((dataItem) => dataItem.type !== "app-events");
    const titleKey = hasOffers ? "ProductPage.Section.AppEventsAndOffers.Title" : "ProductPage.Section.AppEvents.Title";
    const shelfTitle = objectGraph.loc.string(titleKey);
    const shelfId = hasOffers ? "EventsAndOffers" : "Events";
    metricsHelpersLocation.pushContentLocation(objectGraph, {
        pageInformation: shelfMetrics.metricsPageInformation,
        locationTracker: shelfMetrics.locationTracker,
        recoMetricsData: metricsOptions.recoMetricsData,
        targetType: "swoosh",
        id: shelfId,
        idType: "none",
    }, shelfTitle);
    const displayableAppEvents = appPromotionsCommon.appPromotionsFromData(objectGraph, eventAndOffersData, data, !hasOffers, false, metricsOptions, false, false, false, false);
    if (isSome(displayableAppEvents.nextAppEventPromotionStartDate)) {
        refresh.addNextPreferredContentRefreshDate(displayableAppEvents.nextAppEventPromotionStartDate, refreshController);
    }
    const appEvents = displayableAppEvents.appPromotions;
    if (appEvents.length === 0) {
        metricsHelpersLocation.popLocation(metricsOptions.locationTracker);
        return null;
    }
    const shelf = appEventsShelf(objectGraph, appEvents, shelfTitle);
    shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "swoosh", shelfId, "none", metricsOptions.recoMetricsData);
    metricsHelpersLocation.popLocation(metricsOptions.locationTracker);
    metricsHelpersLocation.nextPosition(metricsOptions.locationTracker);
    return shelf;
}
/**
 * Builder for the app events shelf, in a today article.
 * @param objectGraph
 * @param dataItems The array of data blobs
 * @param metricsPageInformation The metrics page information.
 * @param metricsLocationTracker The metrics location tracker.
 * @param context
 * @return The built shelf or null
 */
export function appEventsShelfForArticle(objectGraph, dataItems, metricsPageInformation, metricsLocationTracker, context) {
    if (!appPromotionsCommon.appEventsAreEnabled(objectGraph)) {
        return null;
    }
    if (serverData.isNullOrEmpty(dataItems)) {
        return null;
    }
    const metricsOptions = {
        pageInformation: metricsPageInformation,
        locationTracker: metricsLocationTracker,
    };
    const displayableAppEvents = appPromotionsCommon.appPromotionsFromData(objectGraph, dataItems, null, false, false, metricsOptions, true, true, false, context.allowUnpublishedAppEventPreviews);
    if (isSome(displayableAppEvents.nextAppEventPromotionStartDate) && isSome(context === null || context === void 0 ? void 0 : context.refreshController)) {
        refresh.addNextPreferredContentRefreshDate(displayableAppEvents.nextAppEventPromotionStartDate, context === null || context === void 0 ? void 0 : context.refreshController);
    }
    const appEvents = displayableAppEvents.appPromotions;
    if (appEvents.length === 0) {
        return null;
    }
    const shelf = appEventsShelf(objectGraph, appEvents, undefined);
    // Items should be centered in the article context.
    shelf.isHorizontal = false;
    // We don't really need this shelf impression, but without it the contained items
    // won't be impressed as it is a horizontal shelf
    const shelfMetricsOptions = {
        ...metricsOptions,
        id: "",
        kind: null,
        softwareType: null,
        targetType: "swoosh",
        title: "",
    };
    metricsHelpersImpressions.addImpressionFields(objectGraph, shelf, shelfMetricsOptions);
    return shelf;
}
/**
 * Convenience function for creating a shelf from a list of appEvents
 * @param objectGraph
 * @param appEvents The array of app events
 * @param title The title for the shelf
 */
function appEventsShelf(objectGraph, appEvents, title) {
    const shelfType = "appPromotion";
    const shelf = new models.Shelf(shelfType);
    shelf.isHorizontal = true;
    shelf.title = title;
    shelf.items = appEvents;
    return shelf;
}
// endregion
//# sourceMappingURL=app-promotions-shelf.js.map