import * as models from "../../api/models";
import * as appEventDetail from "./app-event-detail";
import * as contingentOfferDetail from "./contingent-offer-detail";
import * as offerItemDetail from "./offer-item-detail";
import * as appEventModel from "./app-event";
import * as contingentOfferModel from "./contingent-offer";
import * as offerItemModel from "./offer-item";
/**
 * @param data The MAPI data to determine the promotion type
 * @returns The App Promotion type, or null if the type is unknown
 */
export function promotionTypeFromData(data) {
    switch (data.type) {
        case "app-events":
            return models.AppPromotionType.AppEvent;
        case "contingent-items":
            return models.AppPromotionType.ContingentOffer;
        case "offer-items":
            return models.AppPromotionType.OfferItem;
        default:
            return null;
    }
}
/**
 * @param data The MAPI data to determine the promotion type
 * @returns The target type for metrics events for the given app promotion
 */
export function metricsTargetTypeFromData(data) {
    switch (data.type) {
        case "app-events":
            return "eventModule";
        case "contingent-items":
            return "module";
        case "offer-items":
            return "module";
        default:
            return null;
    }
}
/**
 * @param data The MAPI data to determine the MetricsKind
 * @returns The metrics kind for metrics events for the given app promotion
 */
export function metricsKindFromData(data) {
    switch (data.type) {
        case "app-events":
            return "inAppEvent";
        case "contingent-items":
            return "contingentPriceOffer";
        case "offer-items":
            return "winbackPriceOffer";
        default:
            return null;
    }
}
/**
 * Creates an app promotion (App Event or Contingent Offer) object or a Date, if the event should not yet be shown, from the given data.
 * @param data The data blob
 * @param parentAppData The related parent app of this app promotion. If not provided will be derived from `data`.
 * @param hideLockupWhenNotInstalled Whether to hide the lockup chin when the app is installed
 * @param includeClickAction Whether to generate a click action for the app promotion
 * @param offerEnvironment The preferred environment for the offer
 * @param offerStyle The preferred style of the offer
 * @param includeCrossLinkTitles Whether the cross link titles will be displayed when the app is installed
 * @param allowMissingParentApp Whether to still create the app event if the parent app is missing
 * @param baseMetricsOptions The base metrics options
 * @param allowEndedEvents Whether events in the past are allowed
 * @param includeLockupClickAction Whether to include the click action for the lockup
 * @param isArcadePage Whether or not this is presented on the Arcade page
 * @param allowUnpublishedAppEventPreviews Whether or not to allow event previews
 * @returns an AppPromotion, or a Date if the event's `promotionStartDate` is in the future.
 */
export function appPromotionOrDateFromData(objectGraph, data, parentAppData, hideLockupWhenNotInstalled, includeClickAction, offerEnvironment, offerStyle, includeCrossLinkTitles, baseMetricsOptions, allowEndedEvents, includeLockupClickAction, editorialKind, isArcadePage, allowUnpublishedAppEventPreviews) {
    const promotionType = promotionTypeFromData(data);
    const promotionBaseMetricsOptions = {
        ...baseMetricsOptions,
        targetType: metricsTargetTypeFromData(data),
    };
    switch (promotionType) {
        case models.AppPromotionType.AppEvent:
            return appEventModel.appEventOrPromotionStartDateFromData(objectGraph, data, parentAppData, hideLockupWhenNotInstalled, includeClickAction, offerEnvironment, offerStyle, includeCrossLinkTitles, promotionBaseMetricsOptions, allowEndedEvents, includeLockupClickAction, editorialKind, isArcadePage, allowUnpublishedAppEventPreviews);
        case models.AppPromotionType.ContingentOffer:
            return contingentOfferModel.contingentOfferFromData(objectGraph, data, parentAppData, offerEnvironment, offerStyle, promotionBaseMetricsOptions, includeLockupClickAction, isArcadePage);
        case models.AppPromotionType.OfferItem:
            return offerItemModel.offerItemFromData(objectGraph, data, parentAppData, offerEnvironment, offerStyle, promotionBaseMetricsOptions, includeLockupClickAction, isArcadePage);
        default:
            return null;
    }
}
/**
 * Create a flow action for navigating to the app promotion detail page.
 * @param data The data blob
 * @param parentAppData The associated parent app data
 * @param appPromotion The source app promotion
 * @param baseMetricsOptions The base metrics options
 * @param animationBehavior The animation behaviour for presenting the modal page
 * @param includeLockupClickAction Whether to generate a click action for the lockup
 * @param referrerData Referrer data from an incoming deep link
 */
export function detailPageFlowActionFromData(objectGraph, data, parentAppData, appPromotion, baseMetricsOptions, animationBehavior, includeLockupClickAction, referrerData) {
    const promotionType = promotionTypeFromData(data);
    const promotionBaseMetricsOptions = {
        ...baseMetricsOptions,
        targetType: metricsTargetTypeFromData(data),
    };
    switch (promotionType) {
        case models.AppPromotionType.AppEvent:
            const appEvent = appPromotion;
            return appEventDetail.appEventDetailPageFlowActionFromData(objectGraph, data, parentAppData, appEvent, promotionBaseMetricsOptions, animationBehavior, includeLockupClickAction, referrerData);
        case models.AppPromotionType.ContingentOffer:
            const contingentOffer = appPromotion;
            return contingentOfferDetail.contingentOfferDetailPageFlowActionFromData(objectGraph, data, parentAppData, contingentOffer, promotionBaseMetricsOptions, animationBehavior, includeLockupClickAction, referrerData);
        case models.AppPromotionType.OfferItem:
            const offerItem = appPromotion;
            return offerItemDetail.offerItemDetailPageFlowActionFromData(objectGraph, data, parentAppData, offerItem, promotionBaseMetricsOptions, animationBehavior, includeLockupClickAction, referrerData);
        default:
            return null;
    }
}
//# sourceMappingURL=app-promotion.js.map