import { isNothing, isSome } from "@jet/environment";
import { unreachable } from "../../foundation/util/errors";
import { isNotEmpty } from "../../foundation/util/array-util";
let suppressedAccessibilityApps = null;
/**
 * Intiailizes `suppressedAccessibilityApps` with whatever is in the bag for `suppressedAccessibilityAppIds`, so that it only needs to be mapped to a Set once.
 */
function initialize(objectGraph) {
    if (suppressedAccessibilityApps !== null) {
        return;
    }
    suppressedAccessibilityApps = new Set();
    for (const appId of objectGraph.bag.suppressedAccessibilityAppIds) {
        suppressedAccessibilityApps.add(appId);
    }
}
/**
 * Returns whether the accessibility support feature is enabled.
 */
export function isProductAccessibilityLabelsEnabled(objectGraph) {
    return (objectGraph.featureFlags.isEnabled("product_accessibility_support_2025A") &&
        objectGraph.bag.enableAppAccessibilityLabels);
}
/**
 * Returns whether we should suppress the accessibility information for a given adamId.
 */
export function shouldSuppressAccessibilityLabelsForAdamId(objectGraph, adamId) {
    initialize(objectGraph);
    return isSome(suppressedAccessibilityApps) && isNotEmpty(adamId) && suppressedAccessibilityApps.has(adamId);
}
/**
 * Returns whether we should suppress the accessibility information for a given bundleId.
 */
export function shouldSuppressAccessibilityLabelsForBundleId(objectGraph, bundleId) {
    initialize(objectGraph);
    // We always want to suppress accessibility information for macOS installers, as accessibility labels don't apply.
    if (bundleId.startsWith("com.apple.InstallAssistant.")) {
        return true;
    }
    return isSome(suppressedAccessibilityApps) && isNotEmpty(bundleId) && suppressedAccessibilityApps.has(bundleId);
}
/**
 * Returns the device family we want to display the accessibility labels for, based on `AppPlatform` for the first set
 * of screenshots we display on the product page.
 */
export function deviceFamilyForAccessibilityLabels(platform) {
    if (isNothing(platform)) {
        return null;
    }
    switch (platform) {
        case "phone":
            return "iphone";
        case "pad":
            return "ipad";
        case "mac":
            return "mac";
        case "tv":
            return "tvos";
        case "vision":
            return "realityDevice";
        case "watch":
            return "watch";
        case "messages":
            // If we are surfacing Messages metadata, we want to hide the labels.
            return null;
        default:
            unreachable(platform);
    }
}
//# sourceMappingURL=accessibility-common.js.map