"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.areEqual = exports.luminanceFrom = exports.dynamicWith = exports.named = exports.rgbWith = exports.htmlWith = exports.Color = void 0;
const optional_1 = require("../types/optional");
// endregion
// region Constructors
// eslint-disable-next-line no-redeclare, @typescript-eslint/no-redeclare
exports.Color = {
    /**
     * Create new `HTMLColor` from hexadecimal string representation.
     *
     * @param hexString - Hexadecimal string representation.
     */
    fromHex(string) {
        if ((0, optional_1.isNothing)(string)) {
            return null;
        }
        return {
            $kind: "html",
            value: string,
        };
    },
    /**
     * Create new `RBGColor` with RGB components and opacity value.
     *
     * @param red - Red color value.
     * @param green - Green color value.
     * @param blue - Blue color value.
     * @param alpha - Opacity value.
     */
    fromRGB(red, green, blue, alpha = 1.0) {
        const newColor = {
            $kind: "rgb",
            red: red,
            green: green,
            blue: blue,
            alpha: alpha,
        };
        return newColor;
    },
    /**
     * Create new named color using the color name.
     *
     * @param name - The name of the color.
     */
    named(name) {
        const newColor = {
            $kind: "named",
            name: name,
        };
        return newColor;
    },
    /**
     * Create new dynamic color with light and dark color variants.
     *
     * @param lightColor - The light color variant.
     * @param lightHighContrastColor - The light hight-contrast color variant.
     * @param darkColor - The dark color variant.
     * @param darkHighContrastColor - The dark hight-contrast color variant.
     */
    dynamicWith(lightColor, lightHighContrastColor, darkColor, darkHighContrastColor) {
        const newColor = {
            $kind: "dynamic",
            lightColor: lightColor,
            lightHighContrastColor: lightHighContrastColor,
            darkColor: darkColor,
            darkHighContrastColor: darkHighContrastColor,
        };
        return newColor;
    },
    // endregion
    // region Properties
    /**
     * Get the luminance of the color.
     *
     * @param rgbColor - The RGB color to get luminance for.
     */
    luminanceFrom(rgbColor) {
        // Note: This is lifted from UIColor_Private
        // Using RGB color components, calculates and returns (0.2126 * r) + (0.7152 * g) + (0.0722 * b).
        return rgbColor.red * 0.2126 + rgbColor.green * 0.7152 + rgbColor.blue * 0.0722;
    },
    // endregion
    // region Identity
    /**
     * Compare two colors for equality.
     *
     * @param color1 - Left hand side color to compare.
     * @param color2 - Right hand side color to compare.
     * @returns A Boolean indicating whether the colors are equal.
     */
    areEqual(color1, color2) {
        if ((0, optional_1.isNothing)(color1)) {
            return (0, optional_1.isNothing)(color2);
        }
        else if ((0, optional_1.isNothing)(color2)) {
            return (0, optional_1.isNothing)(color1);
        }
        const kind1 = color1.$kind;
        const kind2 = color2.$kind;
        if (kind1 === "named" && kind2 === "named") {
            const namedColor1 = color1;
            const namedColor2 = color2;
            return namedColor1.name === namedColor2.name;
        }
        else if (kind1 === "rgb" && kind2 === "rgb") {
            const rgbColor1 = color1;
            const rgbColor2 = color2;
            return (rgbColor1.red === rgbColor2.red &&
                rgbColor1.green === rgbColor2.green &&
                rgbColor1.blue === rgbColor2.blue &&
                rgbColor1.alpha === rgbColor2.alpha);
        }
        else if (kind1 === "dynamic" && kind2 === "dynamic") {
            const dynamicColor1 = color1;
            const dynamicColor2 = color2;
            return (exports.Color.areEqual(dynamicColor1.lightColor, dynamicColor2.lightColor) &&
                exports.Color.areEqual(dynamicColor1.lightHighContrastColor, dynamicColor2.lightHighContrastColor) &&
                exports.Color.areEqual(dynamicColor1.darkColor, dynamicColor2.darkColor) &&
                exports.Color.areEqual(dynamicColor1.darkHighContrastColor, dynamicColor2.darkHighContrastColor));
        }
        else {
            return false;
        }
    },
};
/**
 * Create new `HTMLColor` from hexadecimal string representation.
 *
 * @param hexString - Hexadecimal string representation.
 *
 * @deprecated This symbol has been moved to `Color.fromHex` and will be removed
 * in the future.
 */
const htmlWith = exports.Color.fromHex;
exports.htmlWith = htmlWith;
/**
 * Create new `RBGColor` with RGB components and opacity value.
 *
 * @param red - Red color value.
 * @param green - Green color value.
 * @param blue - Blue color value.
 * @param alpha - Opacity value.
 *
 * @deprecated This symbol has been moved to `Color.fromRGB` and will be removed
 * in the future.
 */
const rgbWith = exports.Color.fromRGB;
exports.rgbWith = rgbWith;
/**
 * Create new named color using the color name.
 *
 * @param name - The name of the color.
 *
 * @deprecated This symbol has been moved to `Color.named` and will be removed
 * in the future.
 */
const named = exports.Color.named;
exports.named = named;
/**
 * Create new dynamic color with light and dark color variants.
 *
 * @param lightColor - The light color variant.
 * @param lightHighContrastColor - The light hight-contrast color variant.
 * @param darkColor - The dark color variant.
 * @param darkHighContrastColor - The dark hight-contrast color variant.
 *
 * @deprecated This symbol has been moved to `Color.dynamicWith` and will be removed
 * in the future.
 */
const dynamicWith = exports.Color.dynamicWith;
exports.dynamicWith = dynamicWith;
/**
 * Get the luminance of the color.
 *
 * @param rgbColor - The RGB color to get luminance for.
 *
 * @deprecated This symbol has been moved to `Color.luminanceFrom` and will be removed
 * in the future.
 */
const luminanceFrom = exports.Color.luminanceFrom;
exports.luminanceFrom = luminanceFrom;
/**
 * Compare two colors for equality.
 *
 * @param color1 - Left hand side color to compare.
 * @param color2 - Right hand side color to compare.
 * @returns A Boolean indicating whether the colors are equal.
 *
 * @deprecated This symbol has been moved to `Color.areEqual` and will be removed
 * in the future.
 */
const areEqual = exports.Color.areEqual;
exports.areEqual = areEqual;
// endregion
//# sourceMappingURL=color.js.map